/*
 * Decompiled with CFR 0.152.
 */
package org.h2gis.functions.spatial.topology;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import org.h2gis.api.AbstractFunction;
import org.h2gis.api.ScalarFunction;
import org.h2gis.utilities.JDBCUtilities;
import org.h2gis.utilities.SFSUtilities;
import org.h2gis.utilities.TableLocation;
import org.h2gis.utilities.TableUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ST_Graph
extends AbstractFunction
implements ScalarFunction {
    public static final String NODES_SUFFIX = "_NODES";
    public static final String EDGES_SUFFIX = "_EDGES";
    public static final String REMARKS = "ST_Graph produces two tables (nodes and edges) from an input table containing\n`LINESTRING`s or `MULTILINESTRING`s in the given column and using the given\ntolerance, and potentially orienting edges by slope. If the input table has\nname `input`, then the output tables are named `input_nodes` and `input_edges`.\nThe nodes table consists of an integer `node_id` and a `POINT` geometry\nrepresenting each node. The edges table is a copy of the input table with three\nextra columns: `edge_id`, `start_node`, and `end_node`. The `start_node` and\n`end_node` correspond to the `node_id`s in the nodes table.\n\nIf the specified geometry column of the input table contains geometries other\nthan `LINESTRING`s, the operation will fail.\n\nA tolerance value may be given to specify the side length of a square envelope\naround each node used to snap together other nodes within the same envelope.\nNote, however, that edge geometries are left untouched. Note also that\ncoordinates within a given tolerance of each other are not necessarily snapped\ntogether. Only the first and last coordinates of a geometry are considered to\nbe potential nodes, and only nodes within a given tolerance of each other are\nsnapped together. The tolerance works only in metric units.\n\nA boolean value may be set to true to specify that edges should be oriented by\nthe z-value of their first and last coordinates (decreasing).\n";
    private static final Logger LOGGER = LoggerFactory.getLogger((String)("gui." + ST_Graph.class));
    public static final String TYPE_ERROR = "Only LINESTRINGs are accepted. Type code: ";
    public static final String ALREADY_RUN_ERROR = "ST_Graph has already been called on table ";

    public ST_Graph() {
        this.addProperty("remarks", REMARKS);
    }

    public String getJavaStaticMethod() {
        return "createGraph";
    }

    public static boolean createGraph(Connection connection, String tableName) throws SQLException {
        return ST_Graph.createGraph(connection, tableName, null);
    }

    public static boolean createGraph(Connection connection, String tableName, String spatialFieldName) throws SQLException {
        return ST_Graph.createGraph(connection, tableName, spatialFieldName, 0.0);
    }

    public static boolean createGraph(Connection connection, String tableName, String spatialFieldName, double tolerance) throws SQLException {
        return ST_Graph.createGraph(connection, tableName, spatialFieldName, tolerance, false);
    }

    public static boolean createGraph(Connection connection, String inputTable, String spatialFieldName, double tolerance, boolean orientBySlope) throws SQLException {
        return ST_Graph.createGraph(connection, inputTable, spatialFieldName, tolerance, orientBySlope, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean createGraph(Connection connection, String inputTable, String spatialFieldName, double tolerance, boolean orientBySlope, boolean deleteTables) throws SQLException {
        if (tolerance < 0.0) {
            throw new IllegalArgumentException("Only positive tolerances are allowed.");
        }
        TableLocation tableName = TableUtilities.parseInputTable((Connection)connection, (String)inputTable);
        TableLocation nodesName = TableUtilities.suffixTableLocation((TableLocation)tableName, (String)NODES_SUFFIX);
        TableLocation edgesName = TableUtilities.suffixTableLocation((TableLocation)tableName, (String)EDGES_SUFFIX);
        boolean isH2 = JDBCUtilities.isH2DataBase((DatabaseMetaData)connection.getMetaData());
        if (deleteTables) {
            Statement stmt = connection.createStatement();
            StringBuilder sb = new StringBuilder("drop table if exists ");
            sb.append(nodesName.toString(isH2)).append(",").append(edgesName.toString(isH2));
            stmt.execute(sb.toString());
            stmt.close();
        } else if (JDBCUtilities.tableExists((Connection)connection, (String)nodesName.getTable()) || JDBCUtilities.tableExists((Connection)connection, (String)edgesName.getTable())) {
            throw new IllegalArgumentException(ALREADY_RUN_ERROR + tableName.getTable());
        }
        int pkIndex = JDBCUtilities.getIntegerPrimaryKey((Connection)connection, (String)tableName.getTable());
        if (pkIndex == 0) {
            throw new IllegalStateException("Table " + tableName.getTable() + " must contain a single integer primary key.");
        }
        DatabaseMetaData md = connection.getMetaData();
        String pkColName = JDBCUtilities.getFieldName((DatabaseMetaData)md, (String)tableName.getTable(), (int)pkIndex);
        Object[] spatialFieldIndexAndName = ST_Graph.getSpatialFieldIndexAndName(connection, tableName, spatialFieldName);
        int spatialFieldIndex = (Integer)spatialFieldIndexAndName[1];
        spatialFieldName = (String)spatialFieldIndexAndName[0];
        ST_Graph.checkGeometryType(connection, tableName, spatialFieldIndex);
        String geomCol = JDBCUtilities.getFieldName((DatabaseMetaData)md, (String)tableName.getTable(), (int)spatialFieldIndex);
        try (Statement st = connection.createStatement();){
            ST_Graph.firstFirstLastLast(st, tableName, pkColName, geomCol, tolerance);
            int srid = SFSUtilities.getSRID((Connection)connection, (TableLocation)tableName, (String)spatialFieldName);
            ST_Graph.makeEnvelopes(st, tolerance, isH2, srid);
            ST_Graph.nodesTable(st, nodesName, tolerance, isH2, srid);
            ST_Graph.edgesTable(st, nodesName, edgesName, tolerance, isH2);
            ST_Graph.checkForNullEdgeEndpoints(st, edgesName);
            if (orientBySlope) {
                ST_Graph.orientBySlope(st, nodesName, edgesName);
            }
        }
        return true;
    }

    private static void checkGeometryType(Connection connection, TableLocation tableName, int spatialFieldIndex) throws SQLException {
        String fieldName = JDBCUtilities.getFieldName((DatabaseMetaData)connection.getMetaData(), (String)tableName.getTable(), (int)spatialFieldIndex);
        int geomType = SFSUtilities.getGeometryType((Connection)connection, (TableLocation)tableName, (String)fieldName);
        if (geomType != 2) {
            throw new IllegalArgumentException(TYPE_ERROR + SFSUtilities.getGeometryTypeNameFromCode((int)geomType));
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Object[] getSpatialFieldIndexAndName(Connection connection, TableLocation tableName, String spatialFieldName) throws SQLException {
        if (spatialFieldName == null) {
            List geomFields = SFSUtilities.getGeometryFields((Connection)connection, (TableLocation)tableName);
            if (!geomFields.isEmpty()) {
                spatialFieldName = (String)geomFields.get(0);
            } else {
                throw new SQLException("Table " + tableName + " does not contain a geometry field.");
            }
        }
        int spatialFieldIndex = -1;
        try (ResultSet columns = connection.getMetaData().getColumns(tableName.getCatalog(null), tableName.getSchema(null), tableName.getTable(), null);){
            while (columns.next()) {
                if (!columns.getString("COLUMN_NAME").equalsIgnoreCase(spatialFieldName)) continue;
                spatialFieldIndex = columns.getRow();
            }
        }
        if (spatialFieldIndex == -1) {
            throw new SQLException("Geometry field " + spatialFieldName + " of table " + tableName + " not found");
        }
        return new Object[]{spatialFieldName, spatialFieldIndex};
    }

    private static String expand(String geom, double tol) {
        return "ST_Expand(" + geom + ", " + tol + ")";
    }

    private static void firstFirstLastLast(Statement st, TableLocation tableName, String pkCol, String geomCol, double tolerance) throws SQLException {
        LOGGER.info("Selecting the first coordinate of the first geometry and the last coordinate of the last geometry...");
        String numGeoms = "ST_NumGeometries(" + geomCol + ")";
        String firstGeom = "ST_GeometryN(" + geomCol + ", 1)";
        String firstPointFirstGeom = "ST_PointN(" + firstGeom + ", 1)";
        String lastGeom = "ST_GeometryN(" + geomCol + ", " + numGeoms + ")";
        String lastPointLastGeom = "ST_PointN(" + lastGeom + ", ST_NumPoints(" + lastGeom + "))";
        st.execute("drop TABLE if exists COORDS");
        if (tolerance > 0.0) {
            st.execute("CREATE TEMPORARY TABLE COORDS AS SELECT " + pkCol + " EDGE_ID, " + firstPointFirstGeom + " START_POINT, " + ST_Graph.expand(firstPointFirstGeom, tolerance) + " START_POINT_EXP, " + lastPointLastGeom + " END_POINT, " + ST_Graph.expand(lastPointLastGeom, tolerance) + " END_POINT_EXP FROM " + tableName);
        } else {
            st.execute("CREATE TEMPORARY TABLE COORDS AS SELECT " + pkCol + " EDGE_ID, " + firstPointFirstGeom + " START_POINT, " + lastPointLastGeom + " END_POINT FROM " + tableName);
        }
    }

    private static void makeEnvelopes(Statement st, double tolerance, boolean isH2, int srid) throws SQLException {
        st.execute("DROP TABLE IF EXISTS PTS;");
        if (tolerance > 0.0) {
            LOGGER.info("Calculating envelopes around coordinates...");
            if (isH2) {
                st.execute("CREATE TEMPORARY TABLE PTS( ID SERIAL PRIMARY KEY, THE_GEOM POINT, AREA POLYGON ) AS SELECT NULL, START_POINT, START_POINT_EXP FROM COORDS UNION ALL SELECT NULL, END_POINT, END_POINT_EXP FROM COORDS;");
                st.execute("CREATE SPATIAL INDEX ON PTS(AREA);");
            } else {
                st.execute("CREATE TEMPORARY TABLE PTS( ID SERIAL PRIMARY KEY, THE_GEOM GEOMETRY(POINT," + srid + "),AREA GEOMETRY(POLYGON, " + srid + ")) ");
                st.execute("INSERT INTO PTS (SELECT (row_number() over())::int , a.THE_GEOM, A.AREA FROM  (SELECT  START_POINT AS THE_GEOM, START_POINT_EXP as AREA FROM COORDS UNION ALL SELECT  END_POINT AS THE_GEOM, END_POINT_EXP as AREA FROM COORDS) as a);");
                st.execute("CREATE INDEX ON PTS USING GIST(AREA);");
            }
        } else {
            LOGGER.info("Preparing temporary nodes table from coordinates...");
            if (isH2) {
                st.execute("CREATE TEMPORARY TABLE PTS( ID SERIAL PRIMARY KEY, THE_GEOM POINT) AS SELECT NULL, START_POINT FROM COORDS UNION ALL SELECT NULL, END_POINT FROM COORDS;");
                st.execute("CREATE SPATIAL INDEX ON PTS(THE_GEOM);");
            } else {
                st.execute("CREATE TEMPORARY TABLE PTS( ID SERIAL PRIMARY KEY, THE_GEOM GEOMETRY(POINT," + srid + "))");
                st.execute("INSERT INTO PTS (SELECT (row_number() over())::int , a.the_geom FROM (SELECT  START_POINT as THE_GEOM FROM COORDS UNION ALL SELECT  END_POINT as THE_GEOM FROM COORDS) as a);");
                st.execute("CREATE INDEX ON PTS USING GIST(THE_GEOM);");
            }
        }
    }

    private static void nodesTable(Statement st, TableLocation nodesName, double tolerance, boolean isH2, int srid) throws SQLException {
        LOGGER.info("Creating the nodes table...");
        if (tolerance > 0.0) {
            if (isH2) {
                st.execute("CREATE TABLE " + nodesName + "(NODE_ID SERIAL PRIMARY KEY, THE_GEOM POINT, EXP POLYGON) AS SELECT NULL, A.THE_GEOM, A.AREA FROM PTS A, PTS B WHERE A.AREA && B.AREA GROUP BY A.ID HAVING A.ID=MIN(B.ID);");
            } else {
                st.execute("CREATE TABLE " + nodesName + "(NODE_ID SERIAL PRIMARY KEY, THE_GEOM GEOMETRY(POINT, " + srid + "), EXP GEOMETRY(POLYGON," + srid + ")) ");
                st.execute("INSERT INTO " + nodesName + " (SELECT (row_number() over())::int , c.the_geom, c.area FROM (SELECT  A.THE_GEOM, A.AREA FROM PTS A, PTS B WHERE A.AREA && B.AREA GROUP BY A.ID HAVING A.ID=MIN(B.ID)) as c);");
            }
        } else if (isH2) {
            st.execute("CREATE TABLE " + nodesName + "(NODE_ID SERIAL PRIMARY KEY, THE_GEOM POINT) AS SELECT NULL, A.THE_GEOM FROM PTS A, PTS B WHERE A.THE_GEOM && B.THE_GEOM AND A.THE_GEOM=B.THE_GEOM GROUP BY A.ID HAVING A.ID=MIN(B.ID);");
        } else {
            st.execute("CREATE TABLE " + nodesName + "(NODE_ID SERIAL PRIMARY KEY, THE_GEOM GEOMETRY(POINT, " + srid + ")) ");
            st.execute("INSERT INTO " + nodesName + " (SELECT (row_number() over())::int , c.the_geom FROM (SELECT A.THE_GEOM FROM PTS A, PTS B WHERE A.THE_GEOM && B.THE_GEOM AND A.THE_GEOM=B.THE_GEOM GROUP BY A.ID HAVING A.ID=MIN(B.ID)) as c);");
        }
    }

    private static void edgesTable(Statement st, TableLocation nodesName, TableLocation edgesName, double tolerance, boolean isH2) throws SQLException {
        LOGGER.info("Creating the edges table...");
        if (tolerance > 0.0) {
            if (isH2) {
                st.execute("CREATE SPATIAL INDEX ON " + nodesName + "(EXP);");
                st.execute("CREATE SPATIAL INDEX ON COORDS(START_POINT_EXP);");
                st.execute("CREATE SPATIAL INDEX ON COORDS(END_POINT_EXP);");
            } else {
                st.execute("CREATE  INDEX ON " + nodesName + " USING GIST(EXP);");
                st.execute("CREATE  INDEX ON COORDS USING GIST(START_POINT_EXP);");
                st.execute("CREATE  INDEX ON COORDS USING GIST(END_POINT_EXP);");
            }
            st.execute("CREATE TABLE " + edgesName + " AS SELECT EDGE_ID, (SELECT NODE_ID FROM " + nodesName + " WHERE " + nodesName + ".EXP && COORDS.START_POINT_EXP LIMIT 1) START_NODE, (SELECT NODE_ID FROM " + nodesName + " WHERE " + nodesName + ".EXP && COORDS.END_POINT_EXP LIMIT 1) END_NODE FROM COORDS;");
            st.execute("ALTER TABLE " + nodesName + " DROP COLUMN EXP;");
        } else {
            if (isH2) {
                st.execute("CREATE SPATIAL INDEX ON " + nodesName + "(THE_GEOM);");
                st.execute("CREATE SPATIAL INDEX ON COORDS(START_POINT);");
                st.execute("CREATE SPATIAL INDEX ON COORDS(END_POINT);");
            } else {
                st.execute("CREATE INDEX ON " + nodesName + " USING GIST(THE_GEOM);");
                st.execute("CREATE INDEX ON COORDS USING GIST(START_POINT);");
                st.execute("CREATE INDEX ON COORDS USING GIST(END_POINT);");
            }
            st.execute("CREATE TABLE " + edgesName + " AS SELECT EDGE_ID, (SELECT NODE_ID FROM " + nodesName + " WHERE " + nodesName + ".THE_GEOM && COORDS.START_POINT AND " + nodesName + ".THE_GEOM=COORDS.START_POINT LIMIT 1) START_NODE, (SELECT NODE_ID FROM " + nodesName + " WHERE " + nodesName + ".THE_GEOM && COORDS.END_POINT AND " + nodesName + ".THE_GEOM=COORDS.END_POINT LIMIT 1) END_NODE FROM COORDS;");
        }
    }

    private static void orientBySlope(Statement st, TableLocation nodesName, TableLocation edgesName) throws SQLException {
        LOGGER.info("Orienting edges by slope...");
        st.execute("UPDATE " + edgesName + " c SET START_NODE=END_NODE,     END_NODE=START_NODE WHERE (SELECT ST_Z(A.THE_GEOM) < ST_Z(B.THE_GEOM) FROM " + nodesName + " A, " + nodesName + " B WHERE C.START_NODE=A.NODE_ID AND C.END_NODE=B.NODE_ID);");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void checkForNullEdgeEndpoints(Statement st, TableLocation edgesName) throws SQLException {
        LOGGER.info("Checking for null edge endpoints...");
        try (ResultSet nullEdges = st.executeQuery("SELECT COUNT(*) FROM " + edgesName + " WHERE START_NODE IS NULL OR END_NODE IS NULL;");){
            nullEdges.next();
            int n = nullEdges.getInt(1);
            if (n > 0) {
                String msg = "There " + (n == 1 ? "is one edge " : "are " + n + " edges ");
                throw new IllegalStateException(msg + "with a null start node or end node. Try using a slightly smaller tolerance.");
            }
        }
    }
}

