/*
*   Class   Stat
*
*   USAGE:  Statistical functions
*
*   WRITTEN BY: Dr Michael Thomas Flanagan
*
*   DATE:    June 2002 as part of Fmath
*   AMENDED: 12 May 2003 Statistics separated out from Fmath as a new class
*   UPDATE:  18 June 2005, 5 January 2006, 25 April 2006, 12, 21 November 2006
*            4 December 2006 (renaming of cfd and pdf methods - older version also retained)
*            31 December 2006, March 2007, 14 April 2007, 19 October 2007, 27 February 2008
*            29 march 2008, 7 April 2008, 29 April 2008 - 13 May 2008, 22-24 May 2008
*
*   DOCUMENTATION:
*   See Michael Thomas Flanagan's Java library on-line web page:
*   http://www.ee.ucl.ac.uk/~mflanaga/java/Stat.html
*   http://www.ee.ucl.ac.uk/~mflanaga/java/
*
*   Copyright (c) 2002 -  2008
*
*   PERMISSION TO COPY:
*   Permission to use, copy and modify this software and its documentation for
*   NON-COMMERCIAL purposes is granted, without fee, provided that an acknowledgement
*   to the author, Michael Thomas Flanagan at www.ee.ucl.ac.uk/~mflanaga, appears in all copies.
*
*   Dr Michael Thomas Flanagan makes no representations about the suitability
*   or fitness of the software for any or for a particular purpose.
*   Michael Thomas Flanagan shall not be liable for any damages suffered
*   as a result of using, modifying or distributing this software or its derivatives.
*
***************************************************************************************/

package flanagan.analysis;

import java.util.*;
import java.math.*;

import flanagan.math.*;
import flanagan.integration.Integration;
import flanagan.integration.IntegralFunction;
import flanagan.plot.PlotGraph;
import flanagan.complex.*;
import flanagan.circuits.Phasor;
import flanagan.roots.*;
import flanagan.io.*;


public class Stat extends ArrayMaths{

        // INSTANCE VARIABLES
        private ArrayMaths amWeights = null;                                    // weights as arrayMaths
        private boolean weightsSupplied = false;                                // = true if weights entered

        private Vector<Object> upperOutlierDetails = new Vector<Object>();      // upper outlier search details
                                                                                // element 0 - number of ouliers   (Integer)
                                                                                // element 1 - outliers   (double[])
                                                                                // element 2 - outlier indices   (int[])
                                                                                // element 3 - array with ouliers removed
        private boolean upperDone = false;                                      // = true when upper oulier search completed even if no upper outliers found
        private Vector<Object> lowerOutlierDetails = new Vector<Object>();      // lower outlier search details
                                                                                // element 0 - number of ouliers   (Integer)
                                                                                // element 1 - outliers   (double[])
                                                                                // element 2 - outlier indices   (int[])
                                                                                // element 3 - array with ouliers removed
        private boolean lowerDone = false;                                      // = true when lower oulier search completed even if no upper outliers found


        // STATIC VARIABLES
        // A small number close to the smallest representable floating point number
        public static final double FPMIN = 1e-300;

        // PRIVATE MEMBERS FOR USE IN GAMMA FUNCTION METHODS AND HISTOGRAM CONSTRUCTION METHODS

        // GAMMA FUNCTIONS
        //  Lanczos Gamma Function approximation - N (number of coefficients -1)
        private static int lgfN = 6;
        //  Lanczos Gamma Function approximation - Coefficients
        private static double[] lgfCoeff = {1.000000000190015, 76.18009172947146, -86.50532032941677, 24.01409824083091, -1.231739572450155, 0.1208650973866179E-2, -0.5395239384953E-5};
        //  Lanczos Gamma Function approximation - small gamma
        private static double lgfGamma = 5.0;
        //  Maximum number of iterations allowed in Incomplete Gamma Function calculations
        private static int igfiter = 1000;
        //  Tolerance used in terminating series in Incomplete Gamma Function calculations
        private static double igfeps = 1e-8;

        // HISTOGRAM CONSTRUCTION
        //  Tolerance used in including an upper point in last histogram bin when it is outside due to rounding erors
        private static double histTol = 1.0001D;

        // CONSTRUCTORS
        private Stat(){
             super();
        }

        public Stat(double[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Double[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(float[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Float[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(long[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Long[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(int[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Integer[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(short[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Short[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(byte[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Byte[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(BigDecimal[] xx){
             super(xx);
        }

        public Stat(BigInteger[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Complex[] xx){
            super(xx);
            this.convertToHighest();
        }

        public Stat(Phasor[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(String[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Object[] xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(Vector<Object> xx){
             super(xx);
             this.convertToHighest();
        }

        public Stat(ArrayList<Object> xx){
             super(xx);
             this.convertToHighest();
        }

        // Convert array to Double if not Complex, Phasor,  BigDecimal or BigInteger
        // Convert to BigDecimal if BigInteger
        // Convert Phasor to Complex
        public void convertToHighest(){
            switch(this.type){
            case 0:
            case 1:
            case 2:
            case 3:
            case 4:
            case 5:
            case 6:
            case 7:
            case 8:
            case 9:
            case 10:
            case 11:
            case 16:
            case 17:
            case 18: Double[] dd = this.getArray_as_Double();
                     this.array.removeAllElements();
                     for(int i=0; i<this.length; i++)this.array.add(dd[i]);
                     double[] ww = new double[this.length];
                     for(int i=0; i<this.length; i++)ww[i]=1.0D;
                     amWeights = new ArrayMaths(ww);
                     this.type = 1;
                     break;
            case 12:
            case 13: BigDecimal[] bd = this.getArray_as_BigDecimal();
                     this.array.removeAllElements();
                     for(int i=0; i<this.length; i++)this.array.add(bd[i]);
                     BigDecimal[] wd = new BigDecimal[this.length];
                     for(int i=0; i<this.length; i++)wd[i]=BigDecimal.ONE;
                     amWeights = new ArrayMaths(wd);
                     this.type = 12;
                     bd = null;
                     break;
            case 14:
            case 15: Complex[] cc = this.getArray_as_Complex();
                     this.array.removeAllElements();
                     for(int i=0; i<this.length; i++)this.array.add(cc[i]);
                     Complex[] wc = new Complex[this.length];
                     for(int i=0; i<this.length; i++)wc[i]=Complex.plusOne();
                     amWeights = new ArrayMaths(wc);
                     this.type = 14;
                     break;
            default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
            }
        }


        // INSTANCE METHODS
        // Enter array of weights
        public void setWeights(double[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Double[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(float[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Float[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(long[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Long[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(int[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Integer[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(short[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Short[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(byte[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Byte[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(BigDecimal[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(BigInteger[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Complex[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Phasor[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Object[] xx){
            if(this.length!=xx.length)throw new IllegalArgumentException("Length of weights array, " + xx.length + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(Vector<Object> xx){
            if(this.length!=xx.size())throw new IllegalArgumentException("Length of weights array, " + xx.size() + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        public void setWeights(ArrayList<Object> xx){
            if(this.length!=xx.size())throw new IllegalArgumentException("Length of weights array, " + xx.size() + ", must be the same as the length of the instance internal array, " + this.length);
            ArrayMaths wm = new ArrayMaths(xx);
            this.convertWeights(wm);
            this.weightsSupplied = true;
        }

        private void convertWeights(ArrayMaths wm){
            switch(this.type){
            case 1: switch(wm.typeIndex()){
                    case 0:
                    case 1:
                    case 2:
                    case 3:
                    case 4:
                    case 5:
                    case 6:
                    case 7:
                    case 8:
                    case 9:
                    case 10:
                    case 11: Double[] w1 = wm.getArray_as_Double();
                             this.amWeights = new ArrayMaths(w1);
                             break;
                    case 12:
                    case 13: BigDecimal[] a2 = this.getArray_as_BigDecimal();
                             for(int i=0; i<this.length; i++)this.array.add(a2[i]);
                             BigDecimal[] w2 = wm.getArray_as_BigDecimal();
                             this.amWeights = new ArrayMaths(w2);
                             a2 = null;
                             w2 = null;
                             break;
                    case 14:
                    case 15: Complex[] a3 = this.getArray_as_Complex();
                             for(int i=0; i<this.length; i++)this.array.add(a3[i]);
                             Complex[] w3 = wm.getArray_as_Complex();
                             this.amWeights = new ArrayMaths(w3);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");

                    }
                    break;
           case 12: switch(wm.typeIndex()){
                    case 0:
                    case 1:
                    case 2:
                    case 3:
                    case 4:
                    case 5:
                    case 6:
                    case 7:
                    case 8:
                    case 9:
                    case 10:
                    case 11: BigDecimal[] w4 = wm.getArray_as_BigDecimal();
                             this.amWeights = new ArrayMaths(w4);
                             w4 = null;
                             break;
                    case 12:
                    case 13: BigDecimal[] w5 = wm.getArray_as_BigDecimal();
                             this.amWeights = new ArrayMaths(w5);
                             w5 = null;
                             break;
                    case 14:
                    case 15: Complex[] a6 = this.getArray_as_Complex();
                             for(int i=0; i<this.length; i++)this.array.add(a6[i]);
                             Complex[] w6 = wm.getArray_as_Complex();
                             this.amWeights = new ArrayMaths(w6);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                    }
                    break;
            case 14: Complex[] a7 = this.getArray_as_Complex();
                    for(int i=0; i<this.length; i++)this.array.add(a7[i]);
                    Complex[] w7 = wm.getArray_as_Complex();
                    this.amWeights = new ArrayMaths(w7);
                    break;
            default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
            }
        }

        // ARITMETIC MEANS (INSTANCE)
        public double mean_as_double(){
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             for(int i=0; i<this.length; i++){
                                mean += dd[i];
                             }
                             mean /= (double)this.length;
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal meanbd = BigDecimal.ZERO;
                             for(int i=0; i<this.length; i++)meanbd = meanbd.add(bd[i]);
                             meanbd = meanbd.divide(new BigDecimal((double)this.length), BigDecimal.ROUND_HALF_UP);
                             mean = meanbd.doubleValue();
                             bd = null;
                             meanbd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public BigDecimal mean_as_BigDecimal(){
                BigDecimal mean = BigDecimal.ZERO;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double meand= 0.0D;
                             for(int i=0; i<this.length; i++)meand += dd[i];
                             meand /= (double)this.length;
                             mean = new BigDecimal(meand);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             for(int i=0; i<this.length; i++)mean = mean.add(bd[i]);
                             mean = mean.divide(new BigDecimal((double)this.length), BigDecimal.ROUND_HALF_UP);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public Complex mean_as_Complex(){
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double meand= 0.0D;
                             for(int i=0; i<this.length; i++)meand += dd[i];
                             meand /= (double)this.length;
                             mean = new Complex(meand);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal meanbd = BigDecimal.ZERO;
                             for(int i=0; i<this.length; i++)meanbd = meanbd.add(bd[i]);
                             meanbd = meanbd.divide(new BigDecimal((double)this.length), BigDecimal.ROUND_HALF_UP);
                             mean = new Complex(meanbd.doubleValue());
                             bd = null;
                             meanbd = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             for(int i=0; i<this.length; i++)mean = mean.plus(cc[i]);
                             mean = mean.over(new Complex((double)this.length));
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        // CONVERSION OF WEIGHTING FACTORS (INSTANCE)
        // Converts weighting facors Wi to wi, i.e. to 1/sqrt(Wi)
        public void convertBigWtoLittleW(){
            if(!this.weightsSupplied){
                System.out.println("convertBigWtoLittleW: no weights have been supplied - all weights set to unity");
            }
            else{
                amWeights = amWeights.oneOverSqrt();
            }
        }

        // WEIGHTED ARITMETIC MEANS (INSTANCE)
        public double weightedMean_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("weightedMean_as_double: no weights supplied - unweighted mean returned");
                    return this.mean_as_double();
            }
            else{
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] wwd = this.amWeights.getArray_as_double();
                             mean = Stat.mean(dd, wwd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = (Stat.mean(bd, wwb)).doubleValue();
                             bd = null;
                             wwb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public BigDecimal weightedMean_as_BigDecimal(){
            if(!this.weightsSupplied){
                    System.out.println("weightedMean_as_BigDecimal: no weights supplied - unweighted mean returned");
                    return this.mean_as_BigDecimal();
            }
            else{
                BigDecimal mean = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = Stat.mean(bd, wwb);
                             bd = null;
                             wwb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public Complex weightedMean_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("weightedMean_as_Complex: no weights supplied - unweighted mean returned");
                    return this.mean_as_Complex();
            }
            else{
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] wwd = this.amWeights.getArray_as_double();
                             mean = new Complex(Stat.mean(dd, wwd));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = new Complex((Stat.mean(bd, wwb)).doubleValue());
                             bd = null;
                             wwb = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             Complex[] wwc = this.amWeights.getArray_as_Complex();
                             mean = Stat.mean(cc, wwc);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        // SUBTRACT AN ARITMETIC MEAN FROM AN ARRAY (INSTANCE)
        public double[] subtractMean_as_double(){
                double[] arrayminus = new double[this.length];
                switch(this.type){
                    case 1:  double meand = this.mean_as_double();
                             ArrayMaths amd = this.minus(meand);
                             arrayminus = amd.getArray_as_double();
                             break;
                    case 12: BigDecimal meanb = this.mean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_double();
                             meanb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }

            public BigDecimal[] subtractMean_as_BigDecimal(){
                BigDecimal[] arrayminus = new BigDecimal[this.length];
                switch(this.type){
                    case 1:
                    case 12: BigDecimal meanb = this.mean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_BigDecimal();
                             meanb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }

            public Complex[] subtractMean_as_Complex(){
                Complex[] arrayminus = new Complex[this.length];
                switch(this.type){
                    case 1:  double meand = this.mean_as_double();
                             ArrayMaths amd = this.minus(meand);
                             arrayminus = amd.getArray_as_Complex();
                             break;
                    case 12: BigDecimal meanb = this.mean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_Complex();
                             meanb = null;
                             break;
                    case 14: Complex meanc = this.mean_as_Complex();
                             ArrayMaths amc = this.minus(meanc);
                             arrayminus = amc.getArray_as_Complex();
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }

        // SUBTRACT AN WEIGHTED ARITMETIC MEAN FROM AN ARRAY (INSTANCE)
        public double[] subtractWeightedMean_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("subtractWeightedMean_as_double: no weights supplied - unweighted values returned");
                    return this.subtractMean_as_double();
            }
            else{
                double[] arrayminus = new double[this.length];
                switch(this.type){
                    case 1:  double meand = this.weightedMean_as_double();
                             ArrayMaths amd = this.minus(meand);
                             arrayminus = amd.getArray_as_double();
                             break;
                    case 12: BigDecimal meanb = this.weightedMean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_double();
                             meanb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }
        }

        public BigDecimal[] subtractWeightedMean_as_BigDecimal(){
            if(!this.weightsSupplied){
                    System.out.println("subtractWeightedMean_as_BigDecimal: no weights supplied - unweighted values returned");
                    return this.subtractMean_as_BigDecimal();
            }
            else{
                BigDecimal[] arrayminus = new BigDecimal[this.length];
                switch(this.type){
                    case 1:
                    case 12: BigDecimal meanb = this.weightedMean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_BigDecimal();
                             meanb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }
        }

        public Complex[] subtractWeightedMean_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("subtractWeightedMean_as_Complex: no weights supplied - unweighted values returned");
                    return this.subtractMean_as_Complex();
            }
            else{
                Complex[] arrayminus = new Complex[this.length];
                switch(this.type){
                    case 1:  double meand = this.weightedMean_as_double();
                             ArrayMaths amd = this.minus(meand);
                             arrayminus = amd.getArray_as_Complex();
                             break;
                    case 12: BigDecimal meanb = this.weightedMean_as_BigDecimal();
                             ArrayMaths amb = this.minus(meanb);
                             arrayminus = amb.getArray_as_Complex();
                             meanb = null;
                             break;
                    case 14: Complex meanc = this.weightedMean_as_Complex();
                             ArrayMaths amc = this.minus(meanc);
                             arrayminus = amc.getArray_as_Complex();
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return arrayminus;
            }
        }


            // GEOMETRIC MEAN(INSTANCE)
            public double geometricMean_as_double(){
                double gmean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             gmean = Stat.geometricMean(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             gmean = Stat.geometricMean(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot  be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");

                }
                return gmean;
            }

            public Complex geometricMean_as_Complex(){
                Complex gmean = Complex.zero();
                switch(this.type){
                    case 1:
                    case 12:
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             gmean = Stat.geometricMean(cc);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return gmean;
            }


        // WEIGHTED GEOMETRIC MEAN(INSTANCE)
        public double weightedGeometricMean_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("weightedGeometricMean_as_double: no weights supplied - unweighted value returned");
                    return this.geometricMean_as_double();
            }
            else{
                double gmean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = this.getArray_as_double();
                             gmean = Stat.geometricMean(dd, ww);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = this.getArray_as_BigDecimal();
                             gmean = Stat.geometricMean(bd, wd);
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot  be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");

                }
                return gmean;
            }
        }

        public Complex weightedGeometricMean_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("weightedGeometricMean_as_Complex: no weights supplied - unweighted value returned");
                    return this.geometricMean_as_Complex();
            }
            else{
                Complex gmean = Complex.zero();
                switch(this.type){
                    case 1:
                    case 12:
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             Complex[] ww = this.getArray_as_Complex();
                             gmean = Stat.geometricMean(cc, ww);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return gmean;
            }
        }

        // HARMONIC MEANS (INSTANCE)
        public double harmonicMean_as_double(){

                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = Stat.harmonicMean(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = (Stat.harmonicMean(bd)).doubleValue();
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;

        }

        public BigDecimal harmonicMean_as_BigDecimal(){

                BigDecimal mean = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.harmonicMean(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;

        }

        public Complex harmonicMean_as_Complex(){

                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = new Complex(Stat.harmonicMean(dd));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = new Complex((Stat.harmonicMean(bd)).doubleValue());
                             bd = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             mean = Stat.harmonicMean(cc);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;

        }

        // WEIGHTED HARMONIC MEANS (INSTANCE)
        public double weightedHarmonicMean_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("weightedHarmonicMean_as_double: no weights supplied - unweighted mean returned");
                    return this.harmonicMean_as_double();
            }
            else{
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] wwd = this.amWeights.getArray_as_double();
                             mean = Stat.harmonicMean(dd, wwd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = (Stat.harmonicMean(bd, wwb)).doubleValue();
                             bd = null;
                             wwb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public BigDecimal weightedHarmonicMean_as_BigDecimal(){
            if(!this.weightsSupplied){
                    System.out.println("weightedHarmonicMean_as_BigDecimal: no weights supplied - unweighted mean returned");
                    return this.harmonicMean_as_BigDecimal();
            }
            else{
                BigDecimal mean = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = Stat.harmonicMean(bd, wwb);
                             bd = null;
                             wwb = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public Complex weightedHarmonicMean_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("weightedHarmonicMean_as_Complex: no weights supplied - unweighted mean returned");
                    return this.harmonicMean_as_Complex();
            }
            else{
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] wwd = this.amWeights.getArray_as_double();
                             mean = new Complex(Stat.harmonicMean(dd, wwd));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wwb = this.amWeights.getArray_as_BigDecimal();
                             mean = new Complex((Stat.harmonicMean(bd, wwb)).doubleValue());
                             bd = null;
                             wwb = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             Complex[] wwc = this.amWeights.getArray_as_Complex();
                             mean = Stat.harmonicMean(cc, wwc);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        // GENERALIZED MEANS [POWER MEANS](INSTANCE)
        public double generalizedMean_as_double(double m){
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = Stat.generalizedMean(dd, m);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.generalizedMean(bd, m);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;

        }


        public double generalizedMean_as_double(BigDecimal m){
                double mean = 0.0D;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.generalizedMean(bd, m);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public Complex generalizedMean_as_Complex(double m){
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = new Complex(Stat.generalizedMean(dd, m));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = new Complex(Stat.generalizedMean(bd, m));
                             bd = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             mean = Stat.generalizedMean(cc, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public Complex generalizedMean_as_Complex(Complex m){
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:
                    case 12:
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             mean = Stat.generalizedMean(cc, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public double generalisedMean_as_double(double m){
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = Stat.generalisedMean(dd, m);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.generalisedMean(bd, m);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }


        public double generalisedMean_as_double(BigDecimal m){
                double mean = 0.0D;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.generalisedMean(bd, m);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public Complex generalisedMean_as_Complex(double m){
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = new Complex(Stat.generalisedMean(dd, m));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = new Complex(Stat.generalisedMean(bd, m));
                             bd = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             mean = Stat.generalisedMean(cc, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public Complex generalisedMean_as_Complex(Complex m){
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:
                    case 12:
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             mean = Stat.generalisedMean(cc, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }


        // WEIGHTED GENERALIZED MEANS [WEIGHTED POWER MEANS](INSTANCE)
        public double weightedGeneralizedMean_as_double(double m){
            if(!this.weightsSupplied){
                    System.out.println("weightedGeneralizedMean_as_double: no weights supplied - unweighted mean returned");
                    return this.generalizedMean_as_double(m);
            }
            else{
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = this.amWeights.getArray_as_double();
                             mean = Stat.generalisedMean(dd, ww, m);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = this.amWeights.getArray_as_BigDecimal();
                             mean = Stat.generalisedMean(bd, wd, m);
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }


        public double weightedGeneralizedMean_as_double(BigDecimal m){
            if(!this.weightsSupplied){
                    System.out.println("weightedGeneralizedMean_as_double: no weights supplied - unweighted mean returned");
                    return this.generalizedMean_as_double(m);
            }
            else{
                double mean = 0.0D;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = this.amWeights.getArray_as_BigDecimal();
                             mean = Stat.generalisedMean(bd, wd, m);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public Complex weightedGeneralizedMean_as_Complex(double m){
            if(!this.weightsSupplied){
                    System.out.println("weightedGeneralizedMean_as_Complex: no weights supplied - unweighted mean returned");
                    return this.generalizedMean_as_Complex(m);
            }
            else{
                Complex mean = Complex.zero();
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = this.amWeights.getArray_as_double();
                             mean = new Complex(Stat.generalisedMean(dd, ww, m));
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = this.amWeights.getArray_as_BigDecimal();
                             mean = new Complex(Stat.generalisedMean(bd, wd, m));
                             bd = null;
                             break;
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             Complex[] cw = this.amWeights.getArray_as_Complex();
                             mean = Stat.generalisedMean(cc, cw, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public Complex weightedGeneralizedMean_as_Complex(Complex m){
            Complex mean = Complex.zero();
            if(!this.weightsSupplied){
                    System.out.println("weightedGeneralizedMean_as_dComplex: no weights supplied - unweighted mean returned");
                    return this.generalizedMean_as_Complex(m);
            }
            else{
                switch(this.type){
                    case 1:
                    case 12:
                    case 14: Complex[] cc = this.getArray_as_Complex();
                             Complex[] cw = this.amWeights.getArray_as_Complex();
                             mean = Stat.generalisedMean(cc, cw, m);
                             break;
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
            }
        }

        public double weightedGeneralisedMean_as_double(double m){
            return this.weightedGeneralizedMean_as_double(m);
        }

        public double weightedGeneralisedMean_as_double(BigDecimal m){
             return this.weightedGeneralizedMean_as_double(m);
        }

        public Complex weightedGeneralisedMean_as_Complex(double m){
              return this.weightedGeneralizedMean_as_Complex(m);
        }

        public Complex weightedGeneralisedMean_as_Complex(Complex m){
              return this.weightedGeneralizedMean_as_Complex(m);
        }

        // INTERQUARTILE MEANS (INSTANCE)
        public double interQuartileMean_as_double(){
                double mean = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             mean = Stat.interQuartileMean(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = (Stat.interQuartileMean(bd)).doubleValue();
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex interquartile mean is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        public BigDecimal interQuartileMean_as_BigDecimal(){
                BigDecimal mean = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             mean = Stat.interQuartileMean(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex interquartile mean is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return mean;
        }

        // MEDIAN VALUE(INSTANCE)
        public double median_as_double(){
                double median = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             median = Stat.median(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             median = Stat.median(bd).doubleValue();
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex median value not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return median;
            }

            public BigDecimal median_as_BigDecimal(){
                BigDecimal median = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             median = Stat.median(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex median value not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return median;
            }

            // ROOT MEAN SQUARE  (INSTANCE METHODS)
            public double rms(){
                double rms = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             rms = Stat.rms(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             rms = Stat.rms(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex root mean square is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return rms;
            }

                    // WEIGHTED ROOT MEAN SQUARE  (INSTANCE METHODS)
        public double weightedRms(){
            if(!this.weightsSupplied){
                    System.out.println("weightedRms: no weights supplied - unweighted rms returned");
                    return this.rms();
            }
            else{
                double rms = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = this.amWeights.getArray_as_double();
                             rms = Stat.rms(dd, ww);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = this.amWeights.getArray_as_BigDecimal();
                             rms = Stat.rms(bd, wd);
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex root mean square is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return rms;
            }
        }

            // VARIANCES (INSTANCE METHODS)
            public double variance_as_double(){
                double variance = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             variance = Stat.variance(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             variance = (Stat.variance(bd)).doubleValue();
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return variance;
            }

            public BigDecimal variance_as_BigDecimal(){
                BigDecimal variance = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             variance = Stat.variance(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return variance;
            }

            public Complex variance_as_Complex(){
                Complex variance = Complex.zero();
                Complex[] cc = this.getArray_as_Complex();
                variance = Stat.variance(cc);
                return variance;
            }

        // WEIGHTED VARIANCES (INSTANCE METHODS)
        public double weightedVariance_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("weightedVariance_as_double: no weights supplied - unweighted value returned");
                    return this.variance_as_double();
            }
            else{
                double weightedVariance = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = amWeights.getArray_as_double();
                             weightedVariance = Stat.variance(dd, ww);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = amWeights.getArray_as_BigDecimal();
                             weightedVariance = (Stat.variance(bd, wd)).doubleValue();
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return weightedVariance;
            }
        }

        public BigDecimal weightedVariance_as_BigDecimal(){
            if(!this.weightsSupplied){
                    System.out.println("weightedVariance_as_BigDecimal: no weights supplied - unweighted value returned");
                    return this.variance_as_BigDecimal();
            }
            else{
                BigDecimal weightedVariance = BigDecimal.ZERO;
                switch(this.type){
                    case 1:
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = amWeights.getArray_as_BigDecimal();
                             weightedVariance = Stat.variance(bd, wd);
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to BigDecimal");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return weightedVariance;
            }
        }

        public Complex weightedVariance_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("weightedVariance_as_Complex: no weights supplied - unweighted value returned");
                    return this.variance_as_Complex();
            }
            else{
                Complex weightedVariance = Complex.zero();
                Complex[] cc = this.getArray_as_Complex();
                Complex[] wc = amWeights.getArray_as_Complex();
                weightedVariance = Stat.variance(cc, wc);
                return weightedVariance;
            }
        }

            // STANDARD DEVIATIONS (INSTANCE METHODS)
            public double standardDeviation_as_double(){
                double variance = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             variance = Stat.variance(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             variance = (Stat.variance(bd)).doubleValue();
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return Math.sqrt(variance);
            }

            public Complex standardDeviation_as_Complex(){
                Complex variance = Complex.zero();
                Complex[] cc = this.getArray_as_Complex();
                variance = Stat.variance(cc);
                return Complex.sqrt(variance);
            }

        // WEIGHTED STANDARD DEVIATION (INSTANCE METHODS)
        public double weightedStandardDeviation_as_double(){
            if(!this.weightsSupplied){
                    System.out.println("weightedStandardDeviation_as_double: no weights supplied - unweighted value returned");
                    return this.standardDeviation_as_double();
            }
            else{
                double variance = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] ww = amWeights.getArray_as_double();
                             variance = Stat.variance(dd, ww);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] wd = amWeights.getArray_as_BigDecimal();
                             variance = (Stat.variance(bd, wd)).doubleValue();
                             bd = null;
                             wd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex cannot be converted to double");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return Math.sqrt(variance);
            }
        }


        public Complex weightedStandarDeviation_as_Complex(){
            if(!this.weightsSupplied){
                    System.out.println("weightedVariance_as_Complex: no weights supplied - unweighted value returned");
                    return this.standardDeviation_as_Complex();
            }
            else{
                Complex variance = Complex.zero();
                Complex[] cc = this.getArray_as_Complex();
                Complex[] wc = amWeights.getArray_as_Complex();
                variance = Stat.variance(cc, wc);
                return Complex.sqrt(variance);
            }
        }

            // VOLATILITY (INSTANCE METHODS)
            public double volatilityLogChange(){
                double volatilityLogChange = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             volatilityLogChange = Stat.volatilityLogChange(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             volatilityLogChange = Stat.volatilityLogChange(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex volatilty is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return volatilityLogChange;
            }

           public double volatilityPerCentChange(){
                double volatilityPerCentChange = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             volatilityPerCentChange = Stat.volatilityPerCentChange(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             volatilityPerCentChange = Stat.volatilityPerCentChange(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex volatilty is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return volatilityPerCentChange;
            }

            public double coefficientOfVariation(){
                double coefficientOfVariation = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             coefficientOfVariation = Stat.coefficientOfVariation(dd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             coefficientOfVariation = Stat.coefficientOfVariation(bd);
                             bd = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex coefficient of variation is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return coefficientOfVariation;
            }

            public double weightedCoefficientOfVariation(){
                double coefficientOfVariation = 0.0D;
                switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             double[] wd = amWeights.getArray_as_double();
                             coefficientOfVariation = Stat.coefficientOfVariation(dd, wd);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             BigDecimal[] bw = amWeights.getArray_as_BigDecimal();
                             coefficientOfVariation = Stat.coefficientOfVariation(bd, bw);
                             bd = null;
                             bw = null;
                             break;
                    case 14: throw new IllegalArgumentException("Complex coefficient of variation is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                return coefficientOfVariation;
            }

        // OUTLIER DETECTION (INSTANCE)
        // Anscombe test for a upper outlier
        public Vector<Object> upperOutliersAnscombe_as_double(double constant){

            switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             this.upperOutlierDetails = upperOutliersAnscombe(dd, constant);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             Vector<Object> ret = new Vector<Object>();
                             ret = upperOutliersAnscombe(bd, new BigDecimal(constant));
                             this.upperOutlierDetails.add((Integer)ret.elementAt(0));
                             BigDecimal[] bd1 = (BigDecimal[])ret.elementAt(1);
                             ArrayMaths am1 = new ArrayMaths(bd1);
                             this.upperOutlierDetails.add(am1.getArray_as_Double());
                             this.upperOutlierDetails.add((int[])ret.elementAt(2));
                             BigDecimal[] bd2 = (BigDecimal[])ret.elementAt(3);
                             ArrayMaths am2 = new ArrayMaths(bd2);
                             this.upperOutlierDetails.add(am2.getArray_as_Double());
                             break;
                    case 14: throw new IllegalArgumentException("Outlier detection of Complex is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                this.upperDone = true;
                return this.upperOutlierDetails;
        }

        // Identical to upperOutliersAnscombe_as_double(double constant) above
        // retained for compatibility purposes
        public Vector<Object> upperOutliersAnscombe(double constant){
            return this.upperOutliersAnscombe_as_double(constant);
        }

        public Vector<Object> upperOutliersAnscombe_as_BigDecimal(BigDecimal constant){

            switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             Vector<Object> ret = new Vector<Object>();
                             ret = upperOutliersAnscombe(dd, constant.doubleValue());
                             this.upperOutlierDetails.add((Integer)ret.elementAt(0));
                             Double[] dd1 = (Double[])ret.elementAt(1);
                             ArrayMaths am1 = new ArrayMaths(dd1);
                             this.upperOutlierDetails.add(am1.getArray_as_BigDecimal());
                             this.upperOutlierDetails.add((int[])ret.elementAt(2));
                             Double[] dd2 = (Double[])ret.elementAt(3);
                             ArrayMaths am2 = new ArrayMaths(dd2);
                             this.upperOutlierDetails.add(am2.getArray_as_BigDecimal());
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             this.upperOutlierDetails = upperOutliersAnscombe(bd, constant);
                             break;
                    case 14: throw new IllegalArgumentException("Outlier detection of Complex is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                this.upperDone = true;
                return this.upperOutlierDetails;
        }

        public Vector<Object> upperOutliersAnscombe_as_BigDecimal(BigInteger constant){
            return upperOutliersAnscombe_as_BigDecimal(new BigDecimal(constant));
        }


        // Anscombe test for a upper outlier
        public Vector<Object> lowerOutliersAnscombe_as_double(double constant){

            switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             this.lowerOutlierDetails = lowerOutliersAnscombe(dd, constant);
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             Vector<Object> ret = new Vector<Object>();
                             ret = lowerOutliersAnscombe(bd, new BigDecimal(constant));
                             this.lowerOutlierDetails.add((Integer)ret.elementAt(0));
                             BigDecimal[] bd1 = (BigDecimal[])ret.elementAt(1);
                             ArrayMaths am1 = new ArrayMaths(bd1);
                             this.lowerOutlierDetails.add(am1.getArray_as_Double());
                             this.lowerOutlierDetails.add((int[])ret.elementAt(2));
                             BigDecimal[] bd2 = (BigDecimal[])ret.elementAt(3);
                             ArrayMaths am2 = new ArrayMaths(bd2);
                             this.lowerOutlierDetails.add(am2.getArray_as_Double());
                             break;
                    case 14: throw new IllegalArgumentException("Outlier detection of Complex is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                this.lowerDone = true;
                return this.lowerOutlierDetails;
        }

        // Identical to lowerOutliersAnscombe_as_double(double constant) above
        // retained for compatibility purposes
        public Vector<Object> lowerOutliersAnscombe(double constant){
            return this.lowerOutliersAnscombe_as_double(constant);
        }

        public Vector<Object> lowerOutliersAnscombe_as_BigDecimal(BigDecimal constant){

            switch(this.type){
                    case 1:  double[] dd = this.getArray_as_double();
                             Vector<Object> ret = new Vector<Object>();
                             ret = lowerOutliersAnscombe(dd, constant.doubleValue());
                             this.lowerOutlierDetails.add((Integer)ret.elementAt(0));
                             Double[] dd1 = (Double[])ret.elementAt(1);
                             ArrayMaths am1 = new ArrayMaths(dd1);
                             this.lowerOutlierDetails.add(am1.getArray_as_BigDecimal());
                             this.lowerOutlierDetails.add((int[])ret.elementAt(2));
                             Double[] dd2 = (Double[])ret.elementAt(3);
                             ArrayMaths am2 = new ArrayMaths(dd2);
                             this.lowerOutlierDetails.add(am2.getArray_as_BigDecimal());
                             break;
                    case 12: BigDecimal[] bd = this.getArray_as_BigDecimal();
                             this.lowerOutlierDetails = lowerOutliersAnscombe(bd, constant);
                             break;
                    case 14: throw new IllegalArgumentException("Outlier detection of Complex is not supported");
                    default: throw new IllegalArgumentException("This type number, " + this.type +", should not be possible here!!!!");
                }
                this.lowerDone = true;
                return this.lowerOutlierDetails;
        }

        public Vector<Object> lowerOutliersAnscombe_as_BigDecimal(BigInteger constant){
            return lowerOutliersAnscombe_as_BigDecimal(new BigDecimal(constant));
        }





        // STATIC METHODS

        // ARITMETIC MEANS (STATIC)

        // Arithmetic mean of a 1D array of doubles, aa
        public static double mean(double[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=aa[i];
                }
                return sum/((double)n);
        }

        // Arithmetic mean of a 1D array of floats, aa
        public static float mean(float[] aa){
                int n = aa.length;
                float sum=0.0F;
                for(int i=0; i<n; i++){
                        sum+=aa[i];
                }
                return sum/((float)n);
        }

        // Arithmetic mean of a 1D array of int, aa
        public static double mean(int[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(double)aa[i];
                }
                return sum/((double)n);
        }

        // Arithmetic mean of a 1D array of short, aa
        public static double mean(short[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(double)aa[i];
                }
                return sum/((double)n);
        }

        // Arithmetic mean of a 1D array of byte, aa
        public static double mean(byte[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(double)aa[i];
                }
                return sum/((double)n);
        }

        // Arithmetic mean of a 1D array of Complex, aa
        public static Complex mean(Complex[] aa){
                int n = aa.length;
                Complex sum = new Complex(0.0D, 0.0D);
                for(int i=0; i<n; i++){
                        sum = sum.plus(aa[i]);
                }
                return sum.over((double)n);
        }

        // Arithmetic mean of a 1D array of BigDecimal, aa
        public static BigDecimal mean(BigDecimal[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = sum.add(aa[i]);
                }
                return sum.divide(new BigDecimal((double)n), BigDecimal.ROUND_HALF_UP);
        }

        // Arithmetic mean of a 1D array of BigInteger, aa
        public static BigDecimal mean(BigInteger[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal bi = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        bi = new BigDecimal(aa[i]);
                        sum = sum.add(bi);
                }
                bi = null;
                return sum.divide(new BigDecimal((double)n), BigDecimal.ROUND_HALF_UP);
        }


        // CONVERSION OF WEIGHTING FACTORS
        // Converts weighting facors Wi to wi, i.e. to 1/sqrt(Wi)
        public static double[] convertBigWtoLittleW(double[] bigW){
            ArrayMaths am1 = new ArrayMaths(bigW);
            ArrayMaths am2 = am1.oneOverSqrt();
            return am2.getArray_as_double();
        }


        // WEIGHTED ARITHMETIC MEANS (STATIC)
        // Weighted arithmetic mean of a 1D array of doubles, aa
        public static double mean(double[] aa, double[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sumx=0.0D;
                double sumw=0.0D;
                double weight = 0.0D;
                for(int i=0; i<n; i++){
                    weight = 1.0D/(ww[i]*ww[i]);
                    sumx+=aa[i]*weight;
                    sumw+=weight;
                }
                return sumx/sumw;
        }

        // Weighted arithmetic mean of a 1D array of floats, aa
        public static float mean(float[] aa, float[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                float sumx=0.0F;
                float sumw=0.0F;
                float weight = 0.0F;
                for(int i=0; i<n; i++){
                    weight = 1.0F/(ww[i]*ww[i]);
                    sumx+=aa[i]*weight;
                    sumw+=weight;
                }
                return sumx/sumw;
        }

        // Weighted arithmetic mean of a 1D array of Complex, aa
        public static Complex mean(Complex[] aa, Complex[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                Complex sumx=Complex.zero();
                Complex sumw=Complex.zero();
                Complex weight = Complex.zero();
                for(int i=0; i<n; i++){
                    weight = Complex.plusOne().over(ww[i].times(ww[i]));
                    sumx = sumx.plus(aa[i].times(weight));
                    sumw = sumw.plus(weight);
                }
                return sumx.over(sumw);
        }

        // Weighted arithmetic mean of a 1D array of BigDecimal, aa
        public static BigDecimal mean(BigDecimal[] aa, BigDecimal[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                BigDecimal sumx =BigDecimal.ZERO;
                BigDecimal sumw =BigDecimal.ZERO;
                BigDecimal weight = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                    weight = BigDecimal.ONE.divide(ww[i].multiply(ww[i]), BigDecimal.ROUND_HALF_UP);
                    sumx = sumx.add(aa[i].multiply(weight));
                    sumw = sumw.add(weight);
                }
                sumx = sumx.divide(sumw, BigDecimal.ROUND_HALF_UP);
                sumw = null;
                weight = null;
                return sumx;
        }

        // Weighted arithmetic mean of a 1D array of BigInteger, aa
        public static BigDecimal mean(BigInteger[] aa, BigInteger[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                BigDecimal sumx =BigDecimal.ZERO;
                BigDecimal sumw =BigDecimal.ZERO;
                BigDecimal weight = BigDecimal.ZERO;
                BigDecimal aaa = BigDecimal.ZERO;
                BigDecimal www = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                    www = new BigDecimal(ww[i]);
                    weight = BigDecimal.ONE.divide(www.multiply(www), BigDecimal.ROUND_HALF_UP);
                    sumx = sumx.add(aaa.multiply(weight));
                    sumw = sumw.add(weight);
                }
                sumx = sumx.divide(sumw, BigDecimal.ROUND_HALF_UP);
                sumw = null;
                weight = null;
                aaa = null;
                www = null;
                return sumx;
        }

        // SUBTRACT THE MEAN (STATIC)
        // Subtract arithmetic mean of an array from data array elements
        public static double[] subtractMean(double[] array){
            int n = array.length;
            double mean = Stat.mean(array);
            double[] arrayMinusMean = new double[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i] - mean;

            return arrayMinusMean;
        }

        // Subtract arithmetic mean of an array from data array elements
        public static float[] subtractMean(float[] array){
            int n = array.length;
            float mean = Stat.mean(array);
            float[] arrayMinusMean = new float[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i] - mean;

            return arrayMinusMean;
        }


        // Subtract arithmetic mean of an array from data array elements
        public static BigDecimal[] subtractMean(BigDecimal[] array){
            int n = array.length;
            BigDecimal mean = Stat.mean(array);
            BigDecimal[] arrayMinusMean = new BigDecimal[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i].subtract(mean);
            mean = null;
            return arrayMinusMean;
        }

        // Subtract arithmetic mean of an array from data array elements
        public static BigDecimal[] subtractMean(BigInteger[] array){
            int n = array.length;
            BigDecimal mean = Stat.mean(array);
            BigDecimal[] arrayMinusMean = new BigDecimal[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = (new BigDecimal(array[i])).subtract(mean);
            mean = null;
            return arrayMinusMean;
        }

        // Subtract arithmetic mean of an array from data array elements
        public static Complex[] subtractMean(Complex[] array){
            int n = array.length;
            Complex mean = Stat.mean(array);
            Complex[] arrayMinusMean = new Complex[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i].minus(mean);

            return arrayMinusMean;
        }

       // Subtract weighted arirhmetic mean of an array from data array elements
        public static double[] subtractMean(double[] array, double[] weights){
            int n = array.length;
            double mean = Stat.mean(array, weights);
            double[] arrayMinusMean = new double[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i] - mean;

            return arrayMinusMean;
        }

        // Subtract weighted arirhmetic mean of an array from data array elements
        public static float[] subtractMean(float[] array, float[] weights){
            int n = array.length;
            float mean = Stat.mean(array, weights);
            float[] arrayMinusMean = new float[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i] - mean;

            return arrayMinusMean;
        }


        // Subtract weighted arirhmetic mean of an array from data array elements
        public static BigDecimal[] subtractMean(BigDecimal[] array, BigDecimal[] weights){
            int n = array.length;
            BigDecimal mean = Stat.mean(array, weights);
            BigDecimal[] arrayMinusMean = new BigDecimal[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i].subtract(mean);
            mean = null;
            return arrayMinusMean;
        }

        // Subtract weighted arirhmetic mean of an array from data array elements
        public static BigDecimal[] subtractMean(BigInteger[] array, BigInteger[] weights){
            int n = array.length;
            BigDecimal mean = Stat.mean(array, weights);
            BigDecimal[] arrayMinusMean = new BigDecimal[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = (new BigDecimal(array[i])).subtract(mean);
            mean = null;
            return arrayMinusMean;
        }

        // Subtract weighted arirhmetic mean of an array from data array elements
        public static Complex[] subtractMean(Complex[] array, Complex[] weights){
            int n = array.length;
            Complex mean = Stat.mean(array, weights);
            Complex[] arrayMinusMean = new Complex[n];
            for(int i=0; i<n; i++)arrayMinusMean[i] = array[i].minus(mean);

            return arrayMinusMean;
        }

        // GEOMETRIC MEANS (STATIC)

        // Geometric mean of a 1D array of BigDecimal, aa
        public static double geometricMean(BigDecimal[] aa){
                int n = aa.length;
                double sum = 0.0D;
                for(int i=0; i<n; i++)sum += Math.log(aa[i].doubleValue());
                return Math.exp(sum/(double)n);
        }

        // Geometric mean of a 1D array of BigInteger, aa
        public static double geometricMean(BigInteger[] aa){
                int n = aa.length;
                double sum = 0.0D;
                for(int i=0; i<n; i++)sum += Math.log(aa[i].doubleValue());
                return Math.exp(sum/(double)n);
        }

        // Geometric mean of a 1D array of Complex, aa
        public static Complex geometricMean(Complex[] aa){
                int n = aa.length;
                Complex sum = Complex.zero();
                for(int i=0; i<n; i++)sum = sum.plus(Complex.log(aa[i]));
                return Complex.exp(sum.over((double)n));
        }

        // Geometric mean of a 1D array of doubles, aa
        public static double geometricMean(double[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++)sum += Math.log(aa[i]);
                return Math.exp(sum/(double)n);
        }

        // Geometric mean of a 1D array of floats, aa
        public static float geometricMean(float[] aa){
                int n = aa.length;
                float sum=0.0F;
                for(int i=0; i<n; i++)sum += (float)Math.log(aa[i]);
                return (float)Math.exp(sum/(float)n);
        }

          // Weighted geometric mean of a 1D array of Complexs, aa
        public static Complex geometricMean(Complex[] aa, Complex[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                Complex sumW = Complex.zero();
                Complex[] weight = new Complex[n];
                for(int i=0; i<n; i++){
                    weight[i]= Complex.plusOne().over(ww[i].times(ww[i]));
                    sumW = sumW.plus(weight[i]);
                }
                Complex sum = Complex.zero();
                for(int i=0; i<n; i++){
                    sum = sum.plus(Complex.log(aa[i]).times(weight[i]));
                }
                return Complex.exp(sum.over(sumW));
        }

        // Weighted geometric mean of a 1D array of BigDecimal, aa
        public static double geometricMean(BigDecimal[] aa, BigDecimal[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sumW = 0.0D;
                double[] weight = new double[n];

                for(int i=0; i<n; i++){
                    weight[i]=(BigDecimal.ONE.divide(ww[i].multiply(ww[i]), BigDecimal.ROUND_HALF_UP)).doubleValue();
                    sumW += weight[i];
                }
                double sum=0.0D;
                for(int i=0; i<n; i++){
                    sum += Math.log(aa[i].doubleValue())*weight[i];
                }
                return Math.exp(sum/sumW);
        }

        // Weighted geometric mean of a 1D array of BigDecimal, aa
        public static double geometricMean(BigInteger[] aa, BigInteger[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sumW = 0.0D;
                double[] weight = new double[n];
                BigDecimal bd = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                    bd = new BigDecimal(ww[i]);
                    weight[i]=(BigDecimal.ONE.divide(bd.multiply(bd), BigDecimal.ROUND_HALF_UP)).doubleValue();
                    sumW += weight[i];
                }
                double sum=0.0D;
                for(int i=0; i<n; i++){
                    sum += Math.log(aa[i].doubleValue())*weight[i];
                }
                bd = null;
                return Math.exp(sum/sumW);
        }

        // Weighted geometric mean of a 1D array of double, aa
        public static double geometricMean(double[] aa, double[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sumW = 0.0D;
                double[] weight = new double[n];
                for(int i=0; i<n; i++){
                    weight[i]=1.0D/(ww[i]*ww[i]);
                    sumW += weight[i];
                }
                double sum=0.0D;
                for(int i=0; i<n; i++){
                    sum += Math.log(aa[i])*weight[i];
                }
                return Math.exp(sum/sumW);
        }

        // Weighted geometric mean of a 1D array of floats, aa
        public static float geometricMean(float[] aa, float[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                float sumW = 0.0F;
                float[] weight = new float[n];
                for(int i=0; i<n; i++){
                    weight[i]=1.0F/(ww[i]*ww[i]);
                    sumW += weight[i];
                }
                float sum=0.0F;
                for(int i=0; i<n; i++){
                    sum += (float)Math.log(aa[i])*weight[i];
                }
                return (float)Math.exp(sum/sumW);
        }

        // HARMONIC MEANS (STATIC)

        // Harmonic mean of a 1D array of BigDecimal, aa
        public static BigDecimal harmonicMean(BigDecimal[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++)sum = sum.add(BigDecimal.ONE.divide(aa[i], BigDecimal.ROUND_HALF_UP));
                sum = (new BigDecimal((double)n)).divide(sum, BigDecimal.ROUND_HALF_UP);
                return sum;
        }

        // Harmonic mean of a 1D array of BigInteger, aa
        public static BigDecimal harmonicMean(BigInteger[] aa){
                int n = aa.length;
                ArrayMaths am = new ArrayMaths(aa);
                BigDecimal[] bd = am.getArray_as_BigDecimal();
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++)sum = sum.add(BigDecimal.ONE.divide(bd[i], BigDecimal.ROUND_HALF_UP));
                sum = (new BigDecimal((double)n)).divide(sum, BigDecimal.ROUND_HALF_UP);
                bd = null;
                return sum;
        }

        // Harmonic mean of a 1D array of Complex, aa
        public static Complex harmonicMean(Complex[] aa){
                int n = aa.length;
                Complex sum = Complex.zero();
                for(int i=0; i<n; i++)sum = sum.plus(Complex.plusOne().over(aa[i]));
                sum = (new Complex((double)n)).over(sum);
                return sum;
        }

        // Harmonic mean of a 1D array of doubles, aa
        public static double harmonicMean(double[] aa){
                int n = aa.length;
                double sum = 0.0D;
                for(int i=0; i<n; i++)sum += 1.0D/aa[i];
                return (double)n/sum;
        }

        // Harmonic mean of a 1D array of floats, aa
        public static float harmonicMean(float[] aa){
                int n = aa.length;
                float sum = 0.0F;
                for(int i=0; i<n; i++)sum += 1.0F/aa[i];
                return (float)n/sum;
        }

        // Weighted harmonic mean of a 1D array of BigDecimal, aa
        public static BigDecimal harmonicMean(BigDecimal[] aa, BigDecimal[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal sumW = BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                for(int i=0; i<n; i++){
                    weight[i] = BigDecimal.ONE.divide(ww[i].multiply(ww[i]), BigDecimal.ROUND_HALF_UP);
                    sumW = sumW.add(weight[i]);
                }
                for(int i=0; i<n; i++)sum = sum.add(weight[i].divide(aa[i], BigDecimal.ROUND_HALF_UP));
                sum = sumW.divide(sum, BigDecimal.ROUND_HALF_UP);
                sumW = null;
                weight = null;
                return sum;
        }

        // Weighted harmonic mean of a 1D array of BigInteger, aa
        public static BigDecimal harmonicMean(BigInteger[] aa, BigInteger[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                ArrayMaths am = new ArrayMaths(aa);
                BigDecimal[] ba = am.getArray_as_BigDecimal();
                ArrayMaths wm = new ArrayMaths(ww);
                BigDecimal[] bw = am.getArray_as_BigDecimal();
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal sumW = BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                for(int i=0; i<n; i++){
                    weight[i] = BigDecimal.ONE.divide(bw[i].multiply(bw[i]), BigDecimal.ROUND_HALF_UP);
                    sumW = sumW.add(weight[i]);
                }
                for(int i=0; i<n; i++)sum = sum.add(weight[i].divide(ba[i], BigDecimal.ROUND_HALF_UP));
                sum = sumW.divide(sum, BigDecimal.ROUND_HALF_UP);
                sumW = null;
                weight = null;
                ba = null;
                bw = null;
                return sum;
        }

        // Weighted harmonic mean of a 1D array of Complex, aa
        public static Complex harmonicMean(Complex[] aa, Complex[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                Complex sum = Complex.zero();
                Complex sumW = Complex.zero();
                Complex[] weight = new Complex[n];
                for(int i=0; i<n; i++){
                    weight[i] = Complex.plusOne().over(ww[i].times(ww[i]));
                    sumW = sumW.plus(weight[i]);
                }
                for(int i=0; i<n; i++)sum = sum.plus(weight[i].over(aa[i]));
                return sumW.over(sum);
        }

        // Weighted harmonic mean of a 1D array of doubles, aa
        public static double harmonicMean(double[] aa, double[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sum = 0.0D;
                double sumW = 0.0D;
                double[] weight = new double[n];
                for(int i=0; i<n; i++){
                    weight[i]=1.0D/(ww[i]*ww[i]);
                    sumW += weight[i];
                }
                for(int i=0; i<n; i++)sum += weight[i]/aa[i];
                return sumW/sum;
        }

        // Weighted harmonic mean of a 1D array of floats, aa
        public static float harmonicMean(float[] aa, float[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                float sum = 0.0F;
                float sumW = 0.0F;
                float[] weight = new float[n];
                for(int i=0; i<n; i++){
                    weight[i]=1.0F/(ww[i]*ww[i]);
                    sumW += weight[i];
                }
                for(int i=0; i<n; i++)sum += weight[i]/aa[i];
                return sumW/sum;
        }

        // GENERALIZED MEANS [POWER MEANS] (STATIC METHODS)

        // generalized mean of a 1D array of Complex, aa
        public static Complex generalizedMean(Complex[] aa, double m){
                int n = aa.length;
                Complex sum = Complex.zero();
                if(m==0.0D){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(aa[i]));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.pow(aa[i],m));
                    }
                    return Complex.pow(sum.over((double)n), 1.0D/m);
                }
        }

        // generalized mean of a 1D array of Complex, aa
        public static Complex generalizedMean(Complex[] aa, Complex m){
                int n = aa.length;
                Complex sum = Complex.zero();
                if(m.equals(Complex.zero())){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(aa[i]));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.pow(aa[i],m));
                    }
                    return Complex.pow(sum.over((double)n), Complex.plusOne().over(m));
                }
        }

        // generalized mean of a 1D array of BigDecimal, aa
        public static double generalizedMean(BigDecimal[] aa, double m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalizedMean(dd, m);
        }

        // generalized mean of a 1D array of BigDecimal, aa
        public static double generalizedMean(BigDecimal[] aa, BigDecimal m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalizedMean(dd, m.doubleValue());
        }

        // generalized mean of a 1D array of BigInteger, aa
        public static double generalizedMean(BigInteger[] aa, double m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalizedMean(dd, m);
        }

        // generalized mean of a 1D array of BigInteger, aa
        public static double generalizedMean(BigInteger[] aa, BigInteger m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalizedMean(dd, m.doubleValue());
        }

        // generalized mean of a 1D array of doubles, aa
        public static double generalizedMean(double[] aa, double m){
                int n = aa.length;
                double sum=0.0D;
                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += Math.log(aa[i]);
                    }
                    return Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += Math.pow(aa[i],m);
                    }
                    return Math.pow(sum/((double)n), 1.0D/m);
                }
        }

        // generalized mean of a 1D array of floats, aa
        public static float generalizedMean(float[] aa, float m){
                int n = aa.length;
                float sum=0.0F;
                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += (float)Math.log(aa[i]);
                    }
                    return (float)Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += Math.pow(aa[i],m);
                    }
                    return (float)Math.pow(sum/((float)n), 1.0F/m);
                }
        }


        // Generalised mean of a 1D array of Complex, aa
        public static Complex generalisedMean(Complex[] aa, double m){
                int n = aa.length;
                Complex sum = Complex.zero();
                if(m==0.0D){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(aa[i]));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.pow(aa[i],m));
                    }
                    return Complex.pow(sum.over((double)n), 1.0D/m);
                }
        }

        // Generalised mean of a 1D array of Complex, aa
        public static Complex generalisedMean(Complex[] aa, Complex m){
                int n = aa.length;
                Complex sum = Complex.zero();
                if(m.equals(Complex.zero())){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(aa[i]));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.pow(aa[i],m));
                    }
                    return Complex.pow(sum.over((double)n), Complex.plusOne().over(m));
                }
        }

        // Generalised mean of a 1D array of BigDecimal, aa
        public static double generalisedMean(BigDecimal[] aa, double m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalisedMean(dd, m);
        }

        // Generalised mean of a 1D array of BigDecimal, aa
        public static double generalisedMean(BigDecimal[] aa, BigDecimal m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalisedMean(dd, m.doubleValue());
        }

        // Generalised mean of a 1D array of BigInteger, aa
        public static double generalisedMean(BigInteger[] aa, double m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalisedMean(dd, m);
        }

        // Generalised mean of a 1D array of BigInteger, aa
        public static double generalisedMean(BigInteger[] aa, BigInteger m){
                ArrayMaths am = new ArrayMaths(aa);
                double[] dd = am.getArray_as_double();
                return generalisedMean(dd, m.doubleValue());
        }

        // Generalised mean of a 1D array of doubles, aa
        public static double generalisedMean(double[] aa, double m){
                int n = aa.length;
                double sum=0.0D;
                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += Math.log(aa[i]);
                    }
                    return Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += Math.pow(aa[i],m);
                    }
                    return Math.pow(sum/((double)n), 1.0D/m);
                }
        }

        // Generalised mean of a 1D array of floats, aa
        public static float generalisedMean(float[] aa, float m){
                int n = aa.length;
                float sum=0.0F;
                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += (float)Math.log(aa[i]);
                    }
                    return (float)Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += Math.pow(aa[i],m);
                    }
                    return (float)Math.pow(sum/((float)n), 1.0F/m);
                }
        }

        // WEIGHTED GENERALIZED MEANS

        // weighted generalized mean of a 1D array of Complex, aa
        public static Complex generalisedMean(Complex[] aa, Complex[] ww, double m){
                int n = aa.length;
                Complex sum = Complex.zero();

                Complex sumw = Complex.zero();
                Complex[] weight = new Complex[n];
                for(int i=0; i<n; i++){
                    weight[i] = Complex.plusOne().over(ww[i].times(ww[i]));
                    sumw = sumw.plus(weight[i]);
                }

                if(m==0.0D){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(weight[i].times(aa[i])).over(sumw));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum = sum.plus(weight[i].times(Complex.pow(aa[i],m)));
                    }
                    return Complex.pow(sum.over(sumw), 1.0D/m);
                }
        }

        // weighted generalized mean of a 1D array of Complex, aa
        public static Complex generalisedMean(Complex[] aa, Complex[] ww, Complex m){
                int n = aa.length;
                Complex sum = Complex.zero();

                Complex sumw = Complex.zero();
                Complex[] weight = new Complex[n];
                for(int i=0; i<n; i++){
                    weight[i] = Complex.plusOne().over(ww[i].times(ww[i]));
                    sumw = sumw.plus(weight[i]);
                }

                if(m.equals(Complex.zero())){
                    for(int i=0; i<n; i++){
                        sum = sum.plus(Complex.log(weight[i].times(aa[i])).over(sumw));
                    }
                    return Complex.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                         sum = sum.plus(weight[i].times(Complex.pow(aa[i],m)));
                    }
                    return Complex.pow(sum.over(sumw), Complex.plusOne().over(m));
                }
        }

        // weighted generalized mean of a 1D array of BigDecimal, aa
        public static double generalisedMean(BigDecimal[] aa, BigDecimal[] ww, double m){
                ArrayMaths am1 = new ArrayMaths(aa);
                double[] dd = am1.getArray_as_double();
                ArrayMaths am2 = new ArrayMaths(ww);
                double[] wd = am2.getArray_as_double();
                return generalisedMean(dd, wd, m);
        }

        // weighted generalized mean of a 1D array of BigDecimal, aa
        public static double generalisedMean(BigDecimal[] aa, BigDecimal[] ww, BigDecimal m){
                ArrayMaths am1 = new ArrayMaths(aa);
                double[] dd = am1.getArray_as_double();
                ArrayMaths am2 = new ArrayMaths(ww);
                double[] wd = am2.getArray_as_double();
                return generalisedMean(dd, wd, m.doubleValue());
        }

        // weighted generalized mean of a 1D array of BigInteger, aa
        public static double generalisedMean(BigInteger[] aa,  BigInteger[] ww, double m){
                ArrayMaths am1 = new ArrayMaths(aa);
                double[] dd = am1.getArray_as_double();
                ArrayMaths am2 = new ArrayMaths(ww);
                double[] wd = am2.getArray_as_double();
                return generalisedMean(dd, wd, m);
        }

        // weighted generalized mean of a 1D array of BigInteger, aa
        public static double generalisedMean(BigInteger[] aa, BigInteger[] ww, BigInteger m){
                ArrayMaths am1 = new ArrayMaths(aa);
                double[] dd = am1.getArray_as_double();
                ArrayMaths am2 = new ArrayMaths(ww);
                double[] wd = am2.getArray_as_double();
                return generalisedMean(dd, wd, m.doubleValue());
        }

        // weighted generalized mean of a 1D array of doubles, aa
        public static double generalisedMean(double[] aa, double[] ww, double m){
                int n = aa.length;
                double sum=0.0D;
                double sumw=0.0D;
                double[] weight = new double[n];
                for(int i=0; i<n; i++){
                    weight[i] = 1.0/(ww[i]*ww[i]);
                    sumw += weight[i];
                }

                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += Math.log(aa[i]*weight[i]/sumw);
                    }
                    return Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += weight[i]*Math.pow(aa[i],m);
                    }
                    return Math.pow(sum/sumw, 1.0D/m);
                }
        }

        // weighted generalized mean of a 1D array of floats, aa
        public static float generalisedMean(float[] aa, float[] ww, float m){
                int n = aa.length;
                float sum=0.0F;
                float sumw=0.0F;
                float[] weight = new float[n];
                for(int i=0; i<n; i++){
                    weight[i] = 1.0F/(ww[i]*ww[i]);
                    sumw += weight[i];
                }
                if(m==0){
                    for(int i=0; i<n; i++){
                        sum += (float)Math.log(aa[i]);
                    }
                    return (float)Math.exp(sum);
                }
                else{
                    for(int i=0; i<n; i++){
                        sum += Math.pow(aa[i],m);
                    }
                    return (float)Math.pow(sum/sumw, 1.0F/m);
                }
        }


        // weighted generalised mean of a 1D array of Complex, aa
        public static Complex weightedGeneralisedMean(Complex[] aa, Complex[] ww, double m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of Complex, aa
        public static Complex weightedGeneralisedMean(Complex[] aa, Complex[] ww, Complex m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of BigDecimal, aa
        public static double weightedGeneralisedMean(BigDecimal[] aa, BigDecimal[] ww, double m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of BigDecimal, aa
        public static double weightedGeneralisedMean(BigDecimal[] aa, BigDecimal[] ww, BigDecimal m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of BigInteger, aa
        public static double weightedGeneralisedMean(BigInteger[] aa,  BigInteger[] ww, double m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of BigInteger, aa
        public static double weightedGeneralisedMean(BigInteger[] aa, BigInteger[] ww, BigInteger m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of doubles, aa
        public static double weightedGeneralisedMean(double[] aa, double[] ww, double m){
                return generalisedMean(aa, ww, m);
        }

        // weighted generalised mean of a 1D array of floats, aa
        public static float weightedGeneralisedMean(float[] aa, float[] ww, float m){
                return generalisedMean(aa, ww, m);
        }




        // INTERQUARTILE MEANS

        // Interquartile mean of a 1D array of BigDecimal, aa
        public static BigDecimal interQuartileMean(BigDecimal[] aa){
                int n = aa.length;
                if(n<4)throw new IllegalArgumentException("At least 4 array elements needed");
                ArrayMaths am = new ArrayMaths(aa);
                ArrayMaths as = am.sort();
                BigDecimal[] bb = as.getArray_as_BigDecimal();
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=n/4; i<3*n/4; i++)sum = sum.add(bb[i]);
                sum = sum.multiply(new BigDecimal(2.0D/(double)n));
                bb = null;
                return sum;
        }

        // Interquartile mean of a 1D array of BigInteger, aa
        public static BigDecimal interQuartileMean(BigInteger[] aa){
                int n = aa.length;
                if(n<4)throw new IllegalArgumentException("At least 4 array elements needed");
                ArrayMaths am = new ArrayMaths(aa);
                ArrayMaths as = am.sort();
                BigDecimal[] bb = as.getArray_as_BigDecimal();
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=n/4; i<3*n/4; i++)sum = sum.add(bb[i]);
                sum = sum.multiply(new BigDecimal(2.0D/(double)n));
                bb = null;
                return sum;
        }

        // Interquartile mean of a 1D array of doubles, aa
        public static double interQuartileMean(double[] aa){
                int n = aa.length;
                if(n<4)throw new IllegalArgumentException("At least 4 array elements needed");
                double[] bb = Fmath.selectionSort(aa);
                double sum = 0.0D;
                for(int i=n/4; i<3*n/4; i++)sum += bb[i];
                return 2.0*sum/(double)(n);
        }

        // Interquartile mean of a 1D array of floats, aa
        public static float interQuartileMean(float[] aa){
                int n = aa.length;
                if(n<4)throw new IllegalArgumentException("At least 4 array elements needed");
                float[] bb = Fmath.selectionSort(aa);
                float sum = 0.0F;
                for(int i=n/4; i<3*n/4; i++)sum += bb[i];
                return 2.0F*sum/(float)(n);
        }

        // ROOT MEAN SQUARES

       // Root mean square (rms) of a 1D array of doubles, aa
        public static double rms(double[] aa){
                int n = aa.length;
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=aa[i]*aa[i];
                }
                return Math.sqrt(sum/((double)n));
        }

        // Root mean square (rms) of a 1D array of floats, aa
        public static float rms(float[] aa){
                int n = aa.length;
                float sum = 0.0F;
                for(int i=0; i<n; i++){
                        sum+=aa[i]*aa[i];
                }
                sum /= (float)n;

                return (float)Math.sqrt(sum);
        }

        // Root mean square (rms) of a 1D array of BigDecimal, aa
        public static  double rms(BigDecimal[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = sum.add(aa[i].multiply(aa[i]));
                }
                sum  = sum.divide((new BigDecimal(n)), BigDecimal.ROUND_HALF_UP);
                double ret = Math.sqrt(sum.doubleValue());
                sum = null;
                return ret;
        }

        // Root mean square (rms) of a 1D array of BigInteger, aa
        public static  double rms(BigInteger[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal bd = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        bd = new BigDecimal(aa[i]);
                        sum = sum.add(bd.multiply(bd));
                }
                sum  = sum.divide((new BigDecimal(n)), BigDecimal.ROUND_HALF_UP);
                double ret = Math.sqrt(sum.doubleValue());
                bd = null;
                sum = null;
                return ret;
        }

        // WEIGHTED ROOT MEAN SQUARES

        // Weighted root mean square (rms) of a 1D array of doubles, aa
        public static double rms(double[] aa, double[] ww){
                int n = aa.length;
                double sumw =0.0D;
                double[] weight =new double[n];
                for(int i=0; i<n; i++){
                        weight[i] = 1.0D/(ww[i]*ww[i]);
                        sumw += weight[i];
                }
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum += weight[i]*aa[i]*aa[i];
                }
                return Math.sqrt(sum/sumw);
        }

        // Weighted root mean square (rms) of a 1D array of floats, aa
        public static float rms(float[] aa, float[] ww){
                int n = aa.length;
                double sumw =0.0F;
                float[] weight =new float[n];
                for(int i=0; i<n; i++){
                        weight[i] = 1.0F/(ww[i]*ww[i]);
                        sumw += weight[i];
                }
                float sum=0.0F;
                for(int i=0; i<n; i++){
                        sum += weight[i]*aa[i]*aa[i];
                }
                return (float)Math.sqrt(sum/sumw);
        }

        // Weighted root mean square (rms) of a 1D array of BigDecimal, aa
        public static  double rms(BigDecimal[] aa, BigDecimal[] ww){
                int n = aa.length;
                BigDecimal sumw = BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                for(int i=0; i<n; i++){
                        weight[i] = BigDecimal.ONE.divide(ww[i].multiply(ww[i]), BigDecimal.ROUND_HALF_UP);
                        sumw = sumw.add(weight[i]);
                }

                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = sum.add((aa[i].multiply(aa[i])).multiply(weight[i]));
                }
                sum  = sum.divide(sumw, BigDecimal.ROUND_HALF_UP);
                double ret = Math.sqrt(sum.doubleValue());
                sum = null;
                weight = null;
                return ret;
        }

        // Weighted root mean square (rms) of a 1D array of BigInteger, aa
        public static  double rms(BigInteger[] aa, BigInteger[] ww){
                int n = aa.length;
                BigDecimal sumw = BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                BigDecimal w = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        w = new BigDecimal(ww[i]);
                        weight[i] = BigDecimal.ONE.divide(w.multiply(w), BigDecimal.ROUND_HALF_UP);
                        sumw = sumw.add(weight[i]);
                }

                BigDecimal sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        w = new BigDecimal(aa[i]);
                        sum = sum.add((w.multiply(w)).multiply(weight[i]));
                }
                sum  = sum.divide(sumw, BigDecimal.ROUND_HALF_UP);
                double ret = Math.sqrt(sum.doubleValue());
                sum = null;
                weight = null;
                w = null;
                return ret;
        }


        // MEDIANS

        // Median of a 1D array of BigDecimal, aa
        public static BigDecimal median(BigDecimal[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                BigDecimal med = BigDecimal.ZERO;
                ArrayMaths bm = new ArrayMaths(aa);
                ArrayMaths sm = bm.sort();
                BigDecimal[] bb = bm.getArray_as_BigDecimal();
                if(Fmath.isOdd(n)){
                    med = bb[nOverTwo];
                }
                else{
                    med = (bb[nOverTwo-1].add(bb[nOverTwo])).divide(new BigDecimal(2.0D), BigDecimal.ROUND_HALF_UP);
                }
                bb = null;
                return med;
        }

        // Median of a 1D array of BigInteger, aa
        public static BigInteger median(BigInteger[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                BigInteger med = BigInteger.ZERO;
                ArrayMaths bm = new ArrayMaths(aa);
                ArrayMaths sm = bm.sort();
                BigInteger[] bb = bm.getArray_as_BigInteger();
                if(Fmath.isOdd(n)){
                    med = bb[nOverTwo];
                }
                else{
                    med = (bb[nOverTwo-1].add(bb[nOverTwo])).divide(new BigInteger("2"));
                }
                bb = null;
                return med;
        }

        // Median of a 1D array of doubles, aa
        public static double median(double[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                double med = 0.0D;
                double[] bb = Fmath.selectionSort(aa);
                if(Fmath.isOdd(n)){
                    med = bb[nOverTwo];
                }
                else{
                    med = (bb[nOverTwo-1]+bb[nOverTwo])/2.0D;
                }

                return med;
        }


        // Median of a 1D array of floats, aa
        public static float median(float[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                float med = 0.0F;
                float[] bb = Fmath.selectionSort(aa);
                if(Fmath.isOdd(n)){
                    med = bb[nOverTwo];
                }
                else{
                    med = (bb[nOverTwo-1]+bb[nOverTwo])/2.0F;
                }

                return med;
        }

        // Median of a 1D array of int, aa
        public static double median(int[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                double med = 0.0D;
                int[] bb = Fmath.selectionSort(aa);
                if(Fmath.isOdd(n)){
                    med = (double)bb[nOverTwo];
                }
                else{
                    med = (double)(bb[nOverTwo-1]+bb[nOverTwo])/2.0D;
                }

                return med;
        }

        // Median of a 1D array of long, aa
        public static double median(long[] aa){
                int n = aa.length;
                int nOverTwo = n/2;
                double med = 0.0D;
                long[] bb = Fmath.selectionSort(aa);
                if(Fmath.isOdd(n)){
                    med = (double)bb[nOverTwo];
                }
                else{
                    med = (double)(bb[nOverTwo-1]+bb[nOverTwo])/2.0D;
                }

                return med;
        }


        // STANDARD DEVIATIONS

        // Standard deviation of a 1D array of BigDecimals, aa
        public static double standardDeviation(BigDecimal[] aa){
                return Math.sqrt(Stat.variance(aa).doubleValue());
        }

        // Standard deviation of a 1D array of BigIntegers, aa
        public static double standardDeviation(BigInteger[] aa){
                return Math.sqrt(Stat.variance(aa).doubleValue());
        }

        // Standard deviation of a 1D array of Complex, aa
        public static Complex standardDeviation(Complex[] aa){
                return Complex.sqrt(Stat.variance(aa));
        }

        // Standard deviation of a 1D array of doubles, aa
        public static double standardDeviation(double[] aa){
                return Math.sqrt(Stat.variance(aa));
        }

        // Standard deviation of a 1D array of floats, aa
        public static float standardDeviation(float[] aa){
                return (float)Math.sqrt(Stat.variance(aa));
        }

        // Standard deviation of a 1D array of int, aa
        public static double standardDeviation(int[] aa){
                return Math.sqrt(Stat.variance(aa));
        }

        // Standard deviation of a 1D array of long, aa
        public static double standardDeviation(long[] aa){
                return Math.sqrt(Stat.variance(aa));
        }

        // Weighted standard deviation of a 1D array of Complex, aa
        public static Complex standardDeviation(Complex[] aa, Complex[] ww){
                if(aa.length!=ww.length)throw new IllegalArgumentException("length of variable array, " + aa.length + " and length of weight array, " + ww.length + " are different");
                return Complex.sqrt(Stat.variance(aa, ww));
        }

        // Weighted standard deviation of a 1D array of BigDecimal, aa
        public static double standardDeviation(BigDecimal[] aa, BigDecimal[] ww){
                if(aa.length!=ww.length)throw new IllegalArgumentException("length of variable array, " + aa.length + " and length of weight array, " + ww.length + " are different");
                return Math.sqrt(Stat.variance(aa, ww).doubleValue());
        }

        // Weighted standard deviation of a 1D array of BigInteger, aa
        public static double standardDeviation(BigInteger[] aa, BigInteger[] ww){
                if(aa.length!=ww.length)throw new IllegalArgumentException("length of variable array, " + aa.length + " and length of weight array, " + ww.length + " are different");
                return Math.sqrt(Stat.variance(aa, ww).doubleValue());
        }

        // Weighted standard deviation of a 1D array of doubles, aa
        public static double standardDeviation(double[] aa, double[] ww){
                if(aa.length!=ww.length)throw new IllegalArgumentException("length of variable array, " + aa.length + " and length of weight array, " + ww.length + " are different");
                return Math.sqrt(Stat.variance(aa, ww));
        }

        // Weighted standard deviation of a 1D array of floats, aa
        public static float standardDeviation(float[] aa, float[] ww){
                if(aa.length!=ww.length)throw new IllegalArgumentException("length of variable array, " + aa.length + " and length of weight array, " + ww.length + " are different");
                return (float)Math.sqrt(Stat.variance(aa, ww));
        }


        // VOLATILITIES

        // volatility   log  (BigDecimal)
        public static double volatilityLogChange(BigDecimal[] array){
            int n = array.length-1;
            double[] change = new double[n];
            for(int i=0; i<n; i++)change[i] = Math.log((array[i+1].divide(array[i], BigDecimal.ROUND_HALF_UP)).doubleValue());
            return Stat.standardDeviation(change);
        }

        // volatility   log  (BigInteger)
        public static double volatilityLogChange(BigInteger[] array){
            int n = array.length-1;
            double[] change = new double[n];
            for(int i=0; i<n; i++)change[i] = Math.log(((new BigDecimal(array[i+1])).divide( new BigDecimal(array[i]), BigDecimal.ROUND_HALF_UP)).doubleValue());
            return Stat.standardDeviation(change);
        }

        // volatility   log  (doubles)
        public static double volatilityLogChange(double[] array){
            int n = array.length-1;
            double[] change = new double[n];
            for(int i=0; i<n; i++)change[i] = Math.log(array[i+1]/array[i]);
            return Stat.standardDeviation(change);
        }

        // volatility   log  (floats)
        public static float volatilityLogChange(float[] array){
            int n = array.length-1;
            float[] change = new float[n];
            for(int i=0; i<n; i++)change[i] = (float)Math.log(array[i+1]/array[i]);
            return Stat.standardDeviation(change);
        }

        // volatility   percentage (BigDecimal)
        public static double volatilityPerCentChange(BigDecimal[] array){
            int n = array.length-1;
            double[] change = new double[n];
            for(int i=0; i<n; i++)change[i] = ((array[i+1].add(array[i])).multiply((new BigDecimal(100.0D)).divide(array[i], BigDecimal.ROUND_HALF_UP))).doubleValue();
            return Stat.standardDeviation(change);
        }

        // volatility   percentage (Biginteger)
        public static double volatilityPerCentChange(BigInteger[] array){
            int n = array.length-1;
            double[] change = new double[n];
            ArrayMaths am = new ArrayMaths(array);
            BigDecimal[] bd = am.getArray_as_BigDecimal();
            for(int i=0; i<n; i++)change[i] = ((bd[i+1].add(bd[i])).multiply((new BigDecimal(100.0D)).divide(bd[i], BigDecimal.ROUND_HALF_UP))).doubleValue();
            bd = null;
            return Stat.standardDeviation(change);
        }

        // volatility   percentage (double)
        public static double volatilityPerCentChange(double[] array){
            int n = array.length-1;
            double[] change = new double[n];
            for(int i=0; i<n; i++)change[i] = (array[i+1] - array[i])*100.0D/array[i];
            return Stat.standardDeviation(change);
        }

        // volatility   percentage (float)
        public static double volatilityPerCentChange(float[] array){
            int n = array.length-1;
            float[] change = new float[n];
            for(int i=0; i<n; i++)change[i] = (array[i+1] - array[i])*100.0F/array[i];
            return Stat.standardDeviation(change);
        }


        // COEFFICIENT OF VARIATION

        // Coefficient of variation of an array of BigInteger
        public static double coefficientOfVariation(BigInteger[] array){
                  return 100.0D*Stat.standardDeviation(array)/Math.abs(Stat.mean(array).doubleValue());
        }

        // Coefficient of variation of an array of BigDecimals
        public static double coefficientOfVariation(BigDecimal[] array){
                  return 100.0D*Stat.standardDeviation(array)/Math.abs(Stat.mean(array).doubleValue());
        }

        // Coefficient of variation of an array of doubles
        public static double coefficientOfVariation(double[] array){
            return 100.0D*Stat.standardDeviation(array)/Math.abs(Stat.mean(array));
        }

        // Coefficient of variation of an array of float
        public static float coefficientOfVariation(float[] array){
            return 100.0F*Stat.standardDeviation(array)/Math.abs(Stat.mean(array));
        }


        // WEIGHTED COEFFICIENT OF VARIATION

        // Weighted coefficient of variation of an array of BigInteger
        public static double coefficientOfVariation(BigInteger[] array, BigInteger[] weight){
                  return 100.0D*Stat.standardDeviation(array, weight)/Math.abs(Stat.mean(array, weight).doubleValue());
        }

        // Weighted coefficient of variation of an array of BigDecimals
        public static double coefficientOfVariation(BigDecimal[] array, BigDecimal[] weight){
                  return 100.0D*Stat.standardDeviation(array, weight)/Math.abs(Stat.mean(array, weight).doubleValue());
        }

        // Weighted coefficient of variation of an array of doubles
        public static double coefficientOfVariation(double[] array, double[] weight){
            return 100.0D*Stat.standardDeviation(array, weight)/Math.abs(Stat.mean(array, weight));
        }

        // Weighted coefficient of variation of an array of float
        public static float coefficientOfVariation(float[] array, float[] weight){
            return 100.0F*Stat.standardDeviation(array, weight)/Math.abs(Stat.mean(array, weight));
        }

        // VARIANCE
        // Instance method
        // Static methods
        // Variance of a 1D array of BigDecimals, aa
        public static BigDecimal variance(BigDecimal[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal mean  = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = sum.add(aa[i]);
                }
                mean = sum.divide(new BigDecimal((double)n), BigDecimal.ROUND_HALF_UP);
                sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = aa[i].subtract(mean);
                        sum = sum.multiply(sum);
                }
                sum = sum.divide(new BigDecimal((double)(n-1)), BigDecimal.ROUND_HALF_UP);
                mean = null;
                return sum;
        }

        // Variance of a 1D array of BigIntegers, aa
        public static BigDecimal variance(BigInteger[] aa){
                int n = aa.length;
                BigDecimal sum = BigDecimal.ZERO;
                BigDecimal mean  = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = sum.add(new BigDecimal(aa[i]));
                }
                mean = sum.divide(new BigDecimal((double)n), BigDecimal.ROUND_HALF_UP);
                sum = BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sum = new BigDecimal(aa[i]).subtract(mean);
                        sum = sum.multiply(sum);
                }
                sum = sum.divide(new BigDecimal((double)(n-1)), BigDecimal.ROUND_HALF_UP);
                mean = null;
                return sum;
        }

        // Variance of a 1D array of Complex, aa
        public static Complex variance(Complex[] aa){
                int n = aa.length;
                Complex sum = Complex.zero();
                Complex mean  = Complex.zero();
                for(int i=0; i<n; i++){
                        sum = sum.plus(new Complex(aa[i]));
                }
                mean = sum.over(new Complex((double)n));
                sum = Complex.zero();

                for(int i=0; i<n; i++){
                        sum = new Complex(aa[i]).minus(mean);
                        sum = sum.times(sum);
                }
                sum = sum.over(new Complex((double)(n-1)));
                mean = null;
                return sum;
        }

        // Variance of a 1D array of doubles, aa
        public static double variance(double[] aa){
                int n = aa.length;
                double sum=0.0D, mean=0.0D;
                for(int i=0; i<n; i++){
                        sum+=aa[i];
                }
                mean=sum/((double)n);
                sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=Fmath.square(aa[i]-mean);
                }
                return sum/((double)(n-1));
        }

        // Variance of a 1D array of floats, aa
        public static float variance(float[] aa){
                int n = aa.length;
                float sum=0.0F, mean=0.0F;
                for(int i=0; i<n; i++){
                        sum+=aa[i];
                }
                mean=sum/((float)n);
                sum=0.0F;
                for(int i=0; i<n; i++){
                        sum+=Fmath.square(aa[i]-mean);
                }
                return sum/((float)(n-1));
        }

       // Variance of a 1D array of int, aa
        public static double variance(int[] aa){
                int n = aa.length;
                double sum=0.0D, mean=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(double)aa[i];
                }
                mean=sum/((double)n);
                sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=Fmath.square((double)aa[i]-mean);
                }
                return sum/((double)(n-1));
        }

       // Variance of a 1D array of ilong, aa
        public static double variance(long[] aa){
                int n = aa.length;
                double sum=0.0D, mean=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(double)aa[i];
                }
                mean=sum/((double)n);
                sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=Fmath.square((double)aa[i]-mean);
                }
                return sum/((double)(n-1));
        }

        // Weighted variance of a 1D array of doubles, aa
        public static double variance(double[] aa, double[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                double sumx=0.0D, sumw=0.0D, mean=0.0D;
                double[] weight = new double[n];
                for(int i=0; i<n; i++){
                        weight[i]=1.0D/(ww[i]*ww[i]);
                        sumx+=aa[i]*weight[i];
                        sumw+=weight[i];
                }
                mean=sumx/sumw;
                sumx=0.0D;
                for(int i=0; i<n; i++){
                        sumx+=weight[i]*Fmath.square(aa[i]-mean);
                }
                return sumx*(double)(n)/((double)(n-1)*sumw);
        }

        // Weighted variance of a 1D array of floats, aa
        public static float variance(float[] aa, float[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                float sumx=0.0F, sumw=0.0F, mean=0.0F;
                float[] weight = new float[n];
                for(int i=0; i<n; i++){
                        weight[i]=1.0F/(ww[i]*ww[i]);
                        sumx+=aa[i]*weight[i];
                        sumw+=weight[i];
                }
                mean=sumx/sumw;
                sumx=0.0F;
                for(int i=0; i<n; i++){
                        sumx+=weight[i]*Fmath.square(aa[i]-mean);
                }
                return sumx*(float)(n)/((float)(n-1)*sumw);
        }

        // Weighted variance of a 1D array of Complex aa
        public static Complex variance(Complex[] aa, Complex[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                Complex sumx=Complex.zero();
                Complex sumw=Complex.zero();
                Complex mean=Complex.zero();
                Complex[] weight = new Complex[n];
                for(int i=0; i<n; i++){
                        weight[i]=Complex.plusOne().over(ww[i].times(ww[i]));
                        sumx = sumx.plus(aa[i].times(weight[i]));
                        sumw = sumw.plus(weight[i]);
                }
                mean=sumx.over(sumw);
                sumx=Complex.zero();
                for(int i=0; i<n; i++){
                        sumx = sumx.plus(weight[i].times(aa[i].minus(mean)).times(aa[i].minus(mean)));
                }
                return (sumx.times(n)).over((new Complex((double)(n-1)).times(sumw)));
        }

        public static BigDecimal variance(BigDecimal[] aa, BigDecimal[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                BigDecimal sumx=BigDecimal.ZERO;
                BigDecimal sumw=BigDecimal.ZERO;
                BigDecimal mean=BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                for(int i=0; i<n; i++){
                        weight[i]=BigDecimal.ONE.divide(ww[i].multiply(ww[i]), BigDecimal.ROUND_HALF_UP);
                        sumx = sumx.add(aa[i].multiply(weight[i]));
                        sumw = sumw.add(weight[i]);
                }
                mean=sumx.divide(sumw, BigDecimal.ROUND_HALF_UP);
                sumx=BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sumx = sumx.add(weight[i].multiply(aa[i].subtract(mean)).multiply(aa[i].subtract(mean)));
                }
                sumx = (sumx.multiply(new BigDecimal((double)n)).divide(new BigDecimal((double)(n-1)).multiply(sumw), BigDecimal.ROUND_HALF_UP));
                sumw = null;
                mean = null;
                weight = null;
                return sumx;
        }

         public static BigDecimal variance(BigInteger[] aa, BigInteger[] ww){
                int n = aa.length;
                if(n!=ww.length)throw new IllegalArgumentException("length of variable array, " + n + " and length of weight array, " + ww.length + " are different");
                ArrayMaths aab = new ArrayMaths(aa);
                BigDecimal[] aabd = aab.getArray_as_BigDecimal();
                ArrayMaths wwb = new ArrayMaths(ww);
                BigDecimal[] wwbd = wwb.getArray_as_BigDecimal();
                BigDecimal sumx=BigDecimal.ZERO;
                BigDecimal sumw=BigDecimal.ZERO;
                BigDecimal mean=BigDecimal.ZERO;
                BigDecimal[] weight = new BigDecimal[n];
                for(int i=0; i<n; i++){
                        weight[i]=BigDecimal.ONE.divide(wwbd[i].multiply(wwbd[i]), BigDecimal.ROUND_HALF_UP);
                        sumx = sumx.add(aabd[i].multiply(weight[i]));
                        sumw = sumw.add(weight[i]);
                }
                mean=sumx.divide(sumw, BigDecimal.ROUND_HALF_UP);
                sumx=BigDecimal.ZERO;
                for(int i=0; i<n; i++){
                        sumx = sumx.add(weight[i].multiply(aabd[i].subtract(mean)).multiply(aabd[i].subtract(mean)));
                }
                sumx = (sumx.multiply(new BigDecimal((double)n )).divide( new BigDecimal((double)(n-1)).multiply(sumw), BigDecimal.ROUND_HALF_UP));
                aabd = null;
                wwbd = null;
                sumw = null;
                mean = null;
                weight = null;
                return sumx;
        }

        // COVARIANCE

        // Covariance of two 1D arrays of doubles, xx and yy
        public static double covariance(double[] xx, double[] yy){
                int n = xx.length;
                if(n!=yy.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of y array, " + yy.length + " are different");

                double sumx=0.0D, meanx=0.0D;
                double sumy=0.0D, meany=0.0D;
                for(int i=0; i<n; i++){
                        sumx+=xx[i];
                        sumy+=yy[i];
                }
                meanx=sumx/((double)n);
                meany=sumy/((double)n);
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=(xx[i]-meanx)*(yy[i]-meany);
                }
                return sum/((double)(n-1));
        }

        // Covariance of two 1D arrays of floats, xx and yy
        public static float covariance(float[] xx, float[] yy){
                int n = xx.length;
                if(n!=yy.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of y array, " + yy.length + " are different");

                float sumx=0.0F, meanx=0.0F;
                float sumy=0.0F, meany=0.0F;
                for(int i=0; i<n; i++){
                        sumx+=xx[i];
                        sumy+=yy[i];
                }
                meanx=sumx/((float)n);
                meany=sumy/((float)n);
                float sum=0.0F;
                for(int i=0; i<n; i++){
                        sum+=(xx[i]-meanx)*(yy[i]-meany);
                }
                return sum/((float)(n-1));
        }

        // Covariance of two 1D arrays of ints, xx and yy
        public static double covariance(int[] xx, int[] yy){
                int n = xx.length;
                if(n!=yy.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of y array, " + yy.length + " are different");

                double sumx=0.0D, meanx=0.0D;
                double sumy=0.0D, meany=0.0D;
                for(int i=0; i<n; i++){
                        sumx+=(double)xx[i];
                        sumy+=(double)yy[i];
                }
                meanx=sumx/((double)n);
                meany=sumy/((double)n);
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=((double)xx[i]-meanx)*((double)yy[i]-meany);
                }
                return sum/((double)(n-1));
        }

        // Covariance of two 1D arrays of ints, xx and yy
        public static double covariance(long[] xx, long[] yy){
                int n = xx.length;
                if(n!=yy.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of y array, " + yy.length + " are different");

                double sumx=0.0D, meanx=0.0D;
                double sumy=0.0D, meany=0.0D;
                for(int i=0; i<n; i++){
                        sumx+=(double)xx[i];
                        sumy+=(double)yy[i];
                }
                meanx=sumx/((double)n);
                meany=sumy/((double)n);
                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=((double)xx[i]-meanx)*((double)yy[i]-meany);
                }
                return sum/((double)(n-1));
        }

        // Weighted covariance of two 1D arrays of doubles, xx and yy with weights ww
        public static double covariance(double[] xx, double[] yy, double[] ww){
                int n = xx.length;
                if(n!=yy.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of y array, " + yy.length + " are different");
                if(n!=ww.length)throw new IllegalArgumentException("length of x variable array, " + n + " and length of weight array, " + yy.length + " are different");
                double sumx=0.0D, sumy=0.0D, sumw=0.0D, meanx=0.0D, meany=0.0D;
                double[] weight = new double[n];
                for(int i=0; i<n; i++){
                    weight[i]=1.0D/(ww[i]*ww[i]);
                    sumx+=xx[i]*weight[i];
                    sumy+=yy[i]*weight[i];
                    sumw+=weight[i];
                }
                meanx=sumx/sumw;
                meany=sumy/sumw;

                double sum=0.0D;
                for(int i=0; i<n; i++){
                        sum+=weight[i]*(xx[i]-meanx)*(yy[i]-meany);
                }
                return sum*(double)(n)/((double)(n-1)*sumw);
        }


        // CORRELATION COEFFICIENT

        // Calculate correlation coefficient
        // x y data as double
        public static double corrCoeff(double[] xx, double[]yy){

            double temp0 = 0.0D, temp1 = 0.0D;  // working variables
            int nData = xx.length;
            if(yy.length!=nData)throw new IllegalArgumentException("array lengths must be equal");
            int df = nData-1;
            // means
            double mx = 0.0D;
            double my = 0.0D;
            for(int i=0; i<nData; i++){
                mx += xx[i];
                my += yy[i];
            }
            mx /= nData;
            my /= nData;

            // calculate sample variances
            double s2xx = 0.0D;
            double s2yy = 0.0D;
            double s2xy = 0.0D;
            for(int i=0; i<nData; i++){
                s2xx += Fmath.square(xx[i]-mx);
                s2yy += Fmath.square(yy[i]-my);
                s2xy += (xx[i]-mx)*(yy[i]-my);
            }

            // calculate corelation coefficient
            double sampleR = s2xy/Math.sqrt(s2xx*s2yy);

            return sampleR;
        }

        // Calculate correlation coefficient
        // x y data as float
        public static float corrCoeff(float[] x, float[] y){
            int nData = x.length;
            if(y.length!=nData)throw new IllegalArgumentException("array lengths must be equal");
            int n = x.length;
            double[] xx = new double[n];
            double[] yy = new double[n];
            for(int i=0; i<n; i++){
                xx[i] = (double)x[i];
                yy[i] = (double)y[i];
            }
            return (float)Stat.corrCoeff(xx, yy);
        }

        // Calculate correlation coefficient
        // x y data as int
        public static double corrCoeff(int[] x, int[]y){
            int n = x.length;
            if(y.length!=n)throw new IllegalArgumentException("array lengths must be equal");

            double[] xx = new double[n];
            double[] yy = new double[n];
            for(int i=0; i<n; i++){
                xx[i] = (double)x[i];
                yy[i] = (double)y[i];
            }
            return Stat.corrCoeff(xx, yy);
        }

        // Calculate weighted correlation coefficient
        // x y data and weights w as double
        public static double corrCoeff(double[] x, double[]y, double[] w){
            int n = x.length;
            if(y.length!=n)throw new IllegalArgumentException("x and y array lengths must be equal");
            if(w.length!=n)throw new IllegalArgumentException("x and weight array lengths must be equal");

            double sxy = Stat.covariance(x, y, w);
            double sx = Stat.variance(x, w);
            double sy = Stat.variance(y, w);
            return sxy/Math.sqrt(sx*sy);
        }

        // Calculate correlation coefficient
        // Binary data x and y
        // Input is the frequency matrix, F, elements, f(i,j)
        // f(0,0) - element00 - frequency of x and y both = 1
        // f(0,1) - element01 - frequency of x = 0 and y = 1
        // f(1,0) - element10 - frequency of x = 1 and y = 0
        // f(1,1) - element11 - frequency of x and y both = 0
        public static double corrCoeff(int element00, int element01, int element10, int element11){
            return ((double)(element00*element11 - element01*element10))/Math.sqrt((double)((element00+element01)*(element10+element11)*(element00+element10)*(element01+element11)));
        }

        // Calculate correlation coefficient
        // Binary data x and y
        // Input is the frequency matrix, F
        // F(0,0) - frequency of x and y both = 1
        // F(0,1) - frequency of x = 0 and y = 1
        // F(1,0) - frequency of x = 1 and y = 0
        // F(1,1) - frequency of x and y both = 0
        public static double corrCoeff(int[][] freqMatrix){
            double element00 = (double)freqMatrix[0][0];
            double element01 = (double)freqMatrix[0][1];
            double element10 = (double)freqMatrix[1][0];
            double element11 = (double)freqMatrix[1][1];
            return ((element00*element11 - element01*element10))/Math.sqrt(((element00+element01)*(element10+element11)*(element00+element10)*(element01+element11)));
        }

        // Linear correlation coefficient cumulative probablity
        // old name calls renamed method
        public static double linearCorrCoeffProb(double rCoeff, int nu){
            return corrCoeffProb(rCoeff, nu);
        }

        // Linear correlation coefficient cumulative probablity
        public static double corrCoeffProb(double rCoeff, int nu){
            if(Math.abs(rCoeff)>1.0D)throw new IllegalArgumentException("|Correlation coefficient| > 1 :  " + rCoeff);

            // Create instances of the classes holding the function evaluation methods
            CorrCoeff cc = new CorrCoeff();

            // Assign values to constant in the function
            cc.a = ((double)nu - 2.0D)/2.0D;


            double integral = Integration.gaussQuad(cc, Math.abs(rCoeff), 1.0D, 128);

            double preterm = Math.exp(Stat.logGamma((nu+1.0D)/2.0)-Stat.logGamma(nu/2.0D))/Math.sqrt(Math.PI);

            return preterm*integral;
        }

        // Linear correlation coefficient single probablity
        // old name calls renamed method
        public static double linearCorrCoeff(double rCoeff, int nu){
            return Stat.corrCoeffPDF(rCoeff, nu);
        }

        // Linear correlation coefficient single probablity
        public static double corrCoeffPDF(double rCoeff, int nu){
            if(Math.abs(rCoeff)>1.0D)throw new IllegalArgumentException("|Correlation coefficient| > 1 :  " + rCoeff);

            double a = ((double)nu - 2.0D)/2.0D;
            double y = Math.pow((1.0D - Fmath.square(rCoeff)),a);

            double preterm = Math.exp(Stat.logGamma((nu+1.0D)/2.0)-Stat.logGamma(nu/2.0D))/Math.sqrt(Math.PI);

            return preterm*y;
        }

        // Linear correlation coefficient single probablity
        public static double corrCoeffPdf(double rCoeff, int nu){
            if(Math.abs(rCoeff)>1.0D)throw new IllegalArgumentException("|Correlation coefficient| > 1 :  " + rCoeff);

            double a = ((double)nu - 2.0D)/2.0D;
            double y = Math.pow((1.0D - Fmath.square(rCoeff)),a);

            double preterm = Math.exp(Stat.logGamma((nu+1.0D)/2.0)-Stat.logGamma(nu/2.0D))/Math.sqrt(Math.PI);

            return preterm*y;
        }


        // HISTOGRAMS

        // Distribute data into bins to obtain histogram
        // zero bin position and upper limit provided
        public static double[][] histogramBins(double[] data, double binWidth, double binZero, double binUpper){
            int n = 0;              // new array length
            int m = data.length;    // old array length;
            for(int i=0; i<m; i++)if(data[i]<=binUpper)n++;
            if(n!=m){
                double[] newData = new double[n];
                int j = 0;
                for(int i=0; i<m; i++){
                    if(data[i]<=binUpper){
                        newData[j] = data[i];
                        j++;
                    }
                }
                System.out.println((m-n)+" data points, above histogram upper limit, excluded in Stat.histogramBins");
                return histogramBins(newData, binWidth, binZero);
            }
            else{
                 return histogramBins(data, binWidth, binZero);

            }
        }

        // Distribute data into bins to obtain histogram
        // zero bin position provided
        public static double[][] histogramBins(double[] data, double binWidth, double binZero){
            double dmax = Fmath.maximum(data);
            int nBins = (int) Math.ceil((dmax - binZero)/binWidth);
            if(binZero+nBins*binWidth>dmax)nBins++;
            int nPoints = data.length;
            int[] dataCheck = new int[nPoints];
            for(int i=0; i<nPoints; i++)dataCheck[i]=0;
            double[]binWall = new double[nBins+1];
            binWall[0]=binZero;
            for(int i=1; i<=nBins; i++){
                binWall[i] = binWall[i-1] + binWidth;
            }
            double[][] binFreq = new double[2][nBins];
            for(int i=0; i<nBins; i++){
                binFreq[0][i]= (binWall[i]+binWall[i+1])/2.0D;
                binFreq[1][i]= 0.0D;
            }
            boolean test = true;

            for(int i=0; i<nPoints; i++){
                test=true;
                int j=0;
                while(test){
                    if(j==nBins-1){
                        if(data[i]>=binWall[j] && data[i]<=binWall[j+1]*(1.0D + Stat.histTol)){
                            binFreq[1][j]+= 1.0D;
                            dataCheck[i]=1;
                            test=false;
                        }
                    }
                    else{
                        if(data[i]>=binWall[j] && data[i]<binWall[j+1]){
                            binFreq[1][j]+= 1.0D;
                            dataCheck[i]=1;
                            test=false;
                        }
                    }
                    if(test){
                        if(j==nBins-1){
                            test=false;
                        }
                        else{
                            j++;
                        }
                    }
                }
            }
            int nMissed=0;
            for(int i=0; i<nPoints; i++)if(dataCheck[i]==0){
                nMissed++;
                System.out.println("p " + i + " " + data[i] + " " + binWall[0] + " " + binWall[nBins]);
            }
            if(nMissed>0)System.out.println(nMissed+" data points, outside histogram limits, excluded in Stat.histogramBins");
            return binFreq;
        }

        // Distribute data into bins to obtain histogram
        // zero bin position calculated
        public static double[][] histogramBins(double[] data, double binWidth){

            double dmin = Fmath.minimum(data);
            double dmax = Fmath.maximum(data);
            double span = dmax - dmin;
            double binZero = dmin;
            int nBins = (int) Math.ceil(span/binWidth);
            double histoSpan = ((double)nBins)*binWidth;
            double rem = histoSpan - span;
            if(rem>=0){
                binZero -= rem/2.0D;
            }
            else{
                if(Math.abs(rem)/span>histTol){
                    // readjust binWidth
                    boolean testBw = true;
                    double incr = histTol/nBins;
                    int iTest = 0;
                    while(testBw){
                       binWidth += incr;
                       histoSpan = ((double)nBins)*binWidth;
                        rem = histoSpan - span;
                        if(rem<0){
                            iTest++;
                            if(iTest>1000){
                                testBw = false;
                                System.out.println("histogram method could not encompass all data within histogram\nContact Michael thomas Flanagan");
                            }
                        }
                        else{
                            testBw = false;
                        }
                    }
                }
            }

            return Stat.histogramBins(data, binWidth, binZero);
        }

        // Distribute data into bins to obtain histogram and plot histogram
        // zero bin position and upper limit provided
        public static double[][] histogramBinsPlot(double[] data, double binWidth, double binZero, double binUpper){
            String xLegend = null;
            return histogramBinsPlot(data, binWidth, binZero, binUpper, xLegend);
        }

        // Distribute data into bins to obtain histogram and plot histogram
        // zero bin position, upper limit and x-axis legend provided
        public static double[][] histogramBinsPlot(double[] data, double binWidth, double binZero, double binUpper, String xLegend){
            int n = 0;              // new array length
            int m = data.length;    // old array length;
            for(int i=0; i<m; i++)if(data[i]<=binUpper)n++;
            if(n!=m){
                double[] newData = new double[n];
                int j = 0;
                for(int i=0; i<m; i++){
                    if(data[i]<=binUpper){
                        newData[j] = data[i];
                        j++;
                    }
                }
                System.out.println((m-n)+" data points, above histogram upper limit, excluded in Stat.histogramBins");
                return histogramBinsPlot(newData, binWidth, binZero, xLegend);
            }
            else{
                 return histogramBinsPlot(data, binWidth, binZero, xLegend);

            }
        }

        // Distribute data into bins to obtain histogram and plot the histogram
        // zero bin position provided
        public static double[][] histogramBinsPlot(double[] data, double binWidth, double binZero){
            String xLegend = null;
            return histogramBinsPlot(data, binWidth, binZero, xLegend);
        }

        // Distribute data into bins to obtain histogram and plot the histogram
        // zero bin position and x-axis legend provided
        public static double[][] histogramBinsPlot(double[] data, double binWidth, double binZero, String xLegend){
            double[][] results = histogramBins(data, binWidth, binZero);
            int nBins = results[0].length;
            int nPoints = nBins*3+1;
            double[][] cdata = PlotGraph.data(1, nPoints);
            cdata[0][0]=binZero;
            cdata[1][0]=0.0D;
            int k=1;
            for(int i=0; i<nBins; i++){
                cdata[0][k]=cdata[0][k-1];
                cdata[1][k]=results[1][i];
                k++;
                cdata[0][k]=cdata[0][k-1]+binWidth;
                cdata[1][k]=results[1][i];
                k++;
                cdata[0][k]=cdata[0][k-1];
                cdata[1][k]=0.0D;
                k++;
            }

            PlotGraph pg = new PlotGraph(cdata);
            pg.setGraphTitle("Histogram:  Bin Width = "+binWidth);
            pg.setLine(3);
            pg.setPoint(0);
            pg.setYaxisLegend("Frequency");
            if(xLegend!=null)pg.setXaxisLegend(xLegend);
            pg.plot();

            return results;
        }

        // Distribute data into bins to obtain histogram and plot the histogram
        // zero bin position calculated
        public static double[][] histogramBinsPlot(double[] data, double binWidth){
            String xLegend = null;
            return Stat.histogramBinsPlot(data, binWidth, xLegend);
        }

        // Distribute data into bins to obtain histogram and plot the histogram
        // zero bin position calculated, x-axis legend provided
        public static double[][] histogramBinsPlot(double[] data, double binWidth, String xLegend){
            double dmin = Fmath.minimum(data);
            double dmax = Fmath.maximum(data);
            double span = dmax - dmin;
            int nBins = (int) Math.ceil(span/binWidth);
            double rem = ((double)nBins)*binWidth-span;
            double binZero =dmin-rem/2.0D;
            return Stat.histogramBinsPlot(data, binWidth, binZero, xLegend);
        }


        // GAMMA DISTRIBUTION AND GAMMA FUNCTIONS

        // Gamma distribution - three parameter
        // cumulative distribution function
        public static double gammaCDF(double mu, double beta, double gamma, double upperLimit){
            if(upperLimit<mu)throw new IllegalArgumentException("The upper limit, " + upperLimit + "must be equal to or greater than the location parameter, " + mu);
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            double xx = (upperLimit - mu)/beta;
            return regularisedGammaFunction(gamma, xx);
        }

        // Gamma distribution - standard
        // cumulative distribution function
        public static double gammaCDF(double gamma, double upperLimit){
            if(upperLimit<0.0D)throw new IllegalArgumentException("The upper limit, " + upperLimit + "must be equal to or greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            return regularisedGammaFunction(gamma, upperLimit);
        }

        // Gamma distribution - three parameter
        // probablity density function
        public static double gammaPDF(double mu, double beta, double gamma, double x){
            if(x<mu)throw new IllegalArgumentException("The variable, x, " + x + "must be equal to or greater than the location parameter, " + mu);
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            double xx = (x - mu)/beta;
            return Math.pow(xx, gamma-1)*Math.exp(-xx)/(beta*gammaFunction(gamma));
        }

        // Gamma distribution - standard
        // probablity density function
        public static double gammaPDF(double gamma, double x){
            if(x<0.0D)throw new IllegalArgumentException("The variable, x, " + x + "must be equal to or greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            return Math.pow(x, gamma-1)*Math.exp(-x)/gammaFunction(gamma);
        }

        // Gamma distribution - three parameter
        // mean
        public static double gammaMean(double mu, double beta, double gamma){
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            return gamma*beta - mu;
        }

        // Gamma distribution - three parameter
        // mode
        public static double gammaMode(double mu, double beta, double gamma){
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            double mode = Double.NaN;
            if(gamma>=1.0D)mode = (gamma-1.0D)*beta - mu;
            return mode;
        }

        // Gamma distribution - three parameter
        // standard deviation
        public static double gammaStandDev(double mu, double beta, double gamma){
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            return Math.sqrt(gamma)*beta;
        }


        // Returns an array of Gamma random deviates - clock seed
        public static double[] gammaRand(double mu, double beta, double gamma, int n){
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            PsRandom psr = new PsRandom();
            return psr.gammaArray(mu, beta, gamma, n);
        }

        // Returns an array of Gamma random deviates - user supplied seed
        public static double[] gammaRand(double mu, double beta, double gamma, int n, long seed){
            if(beta<=0.0D)throw new IllegalArgumentException("The scale parameter, " + beta + "must be greater than zero");
            if(gamma<=0.0D)throw new IllegalArgumentException("The shape parameter, " + gamma + "must be greater than zero");
            PsRandom psr = new PsRandom(seed);
            return psr.gammaArray(mu, beta, gamma, n);
        }

        // Gamma function
        // Lanczos approximation (6 terms)
        public static double gammaFunction(double x){

                double xcopy = x;
                double first = x + lgfGamma + 0.5;
                double second = lgfCoeff[0];
                double fg = 0.0D;

                if(x>=0.0){
                        if(x>=1.0D && x-(int)x==0.0D){
                                fg = Stat.factorial(x)/x;
                        }
                        else{
                                first = Math.pow(first, x + 0.5)*Math.exp(-first);
                                for(int i=1; i<=lgfN; i++)second += lgfCoeff[i]/++xcopy;
                                fg = first*Math.sqrt(2.0*Math.PI)*second/x;
                        }
                }
                else{
                         fg = -Math.PI/(x*Stat.gamma(-x)*Math.sin(Math.PI*x));
                }
                return fg;
        }

        // Gamma function
        // Lanczos approximation (6 terms)
        // retained for backward compatibity
        public static double gamma(double x){

                double xcopy = x;
                double first = x + lgfGamma + 0.5;
                double second = lgfCoeff[0];
                double fg = 0.0D;

                if(x>=0.0){
                        if(x>=1.0D && x-(int)x==0.0D){
                                fg = Stat.factorial(x)/x;
                        }
                        else{
                                first = Math.pow(first, x + 0.5)*Math.exp(-first);
                                for(int i=1; i<=lgfN; i++)second += lgfCoeff[i]/++xcopy;
                                fg = first*Math.sqrt(2.0*Math.PI)*second/x;
                        }
                }
                else{
                         fg = -Math.PI/(x*Stat.gamma(-x)*Math.sin(Math.PI*x));
                }
                return fg;
        }

        // Return the Lanczos constant gamma
        public static double getLanczosGamma(){
                return Stat.lgfGamma;
        }

        // Return the Lanczos constant N (number of coeeficients + 1)
        public static int getLanczosN(){
                return Stat.lgfN;
        }

        // Return the Lanczos coeeficients
        public static double[] getLanczosCoeff(){
                int n = Stat.getLanczosN()+1;
                double[] coef = new double[n];
                for(int i=0; i<n; i++){
                        coef[i] = Stat.lgfCoeff[i];
                }
                return coef;
        }

        // Return the nearest smallest representable floating point number to zero with mantissa rounded to 1.0
        public static double getFpmin(){
                return Stat.FPMIN;
        }

        // log to base e of the Gamma function
        // Lanczos approximation (6 terms)
        public static double logGammaFunction(double x){
                double xcopy = x;
                double fg = 0.0D;
                double first = x + lgfGamma + 0.5;
                double second = lgfCoeff[0];

                if(x>=0.0){
                        if(x>=1.0 && x-(int)x==0.0){
                                fg = Stat.logFactorial(x)-Math.log(x);
                        }
                        else{
                                first -= (x + 0.5)*Math.log(first);
                                for(int i=1; i<=lgfN; i++)second += lgfCoeff[i]/++xcopy;
                                fg = Math.log(Math.sqrt(2.0*Math.PI)*second/x) - first;
                        }
                }
                else{
                        fg = Math.PI/(Stat.gamma(1.0D-x)*Math.sin(Math.PI*x));

                        if(fg!=1.0/0.0 && fg!=-1.0/0.0){
                                if(fg<0){
                                         throw new IllegalArgumentException("\nThe gamma function is negative");
                                }
                                else{
                                        fg = Math.log(fg);
                                }
                        }
                }
                return fg;
        }

        // log to base e of the Gamma function
        // Lanczos approximation (6 terms)
        // Retained for backward compatibility
        public static double logGamma(double x){
                double xcopy = x;
                double fg = 0.0D;
                double first = x + lgfGamma + 0.5;
                double second = lgfCoeff[0];

                if(x>=0.0){
                        if(x>=1.0 && x-(int)x==0.0){
                                fg = Stat.logFactorial(x)-Math.log(x);
                        }
                        else{
                                first -= (x + 0.5)*Math.log(first);
                                for(int i=1; i<=lgfN; i++)second += lgfCoeff[i]/++xcopy;
                                fg = Math.log(Math.sqrt(2.0*Math.PI)*second/x) - first;
                        }
                }
                else{
                        fg = Math.PI/(Stat.gamma(1.0D-x)*Math.sin(Math.PI*x));

                        if(fg!=1.0/0.0 && fg!=-1.0/0.0){
                                if(fg<0){
                                         throw new IllegalArgumentException("\nThe gamma function is negative");
                                }
                                else{
                                        fg = Math.log(fg);
                                }
                        }
                }
                return fg;
        }



        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        public static double regularisedGammaFunction(double a, double x){
                if(a<0.0D  || x<0.0D)throw new IllegalArgumentException("\nFunction defined only for a >= 0 and x>=0");
                double igf = 0.0D;

                if(x < a+1.0D){
                        // Series representation
                        igf = incompleteGammaSer(a, x);
                }
                else{
                        // Continued fraction representation
                        igf = incompleteGammaFract(a, x);
                }
                return igf;
        }

        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        public static double regularizedGammaFunction(double a, double x){
            return regularisedGammaFunction(a, x);
        }

        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        // Retained for backward compatibility
        public static double regIncompleteGamma(double a, double x){
            return regularisedGammaFunction(a, x);
        }

        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        // Retained for backward compatibility
        public static double incompleteGamma(double a, double x){
            return regularisedGammaFunction(a, x);
        }

        // Complementary Regularised Incomplete Gamma Function Q(a,x) = 1 - P(a,x) = 1 - integral from zero to x of (exp(-t)t^(a-1))dt
        public static double complementaryRegularisedGammaFunction(double a, double x){
                if(a<0.0D  || x<0.0D)throw new IllegalArgumentException("\nFunction defined only for a >= 0 and x>=0");
                double igf = 0.0D;

                if(x!=0.0D){
                        if(x==1.0D/0.0D)
                        {
                                igf=1.0D;
                        }
                        else{
                                if(x < a+1.0D){
                                        // Series representation
                                        igf = 1.0D - incompleteGammaSer(a, x);
                                }
                                else{
                                        // Continued fraction representation
                                        igf = 1.0D - incompleteGammaFract(a, x);
                                }
                        }
                }
                return igf;
        }

        // Complementary Regularised Incomplete Gamma Function Q(a,x) = 1 - P(a,x) = 1 - integral from zero to x of (exp(-t)t^(a-1))dt
        public static double complementaryRegularizedGammaFunction(double a, double x){
            return complementaryRegularisedGammaFunction(a , x);
        }

        // Complementary Regularised Incomplete Gamma Function Q(a,x) = 1 - P(a,x) = 1 - integral from zero to x of (exp(-t)t^(a-1))dt
        // Retained for backward compatibility
        public static double incompleteGammaComplementary(double a, double x){
            return complementaryRegularisedGammaFunction(a , x);
        }

        // Complementary Regularised Incomplete Gamma Function Q(a,x) = 1 - P(a,x) = 1 - integral from zero to x of (exp(-t)t^(a-1))dt
        // Retained for backward compatibility
        public static double regIncompleteGammaComplementary(double a, double x){
                return complementaryRegularisedGammaFunction(a , x);
        }

        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        // Series representation of the function - valid for x < a + 1
        public static double incompleteGammaSer(double a, double x){
                if(a<0.0D  || x<0.0D)throw new IllegalArgumentException("\nFunction defined only for a >= 0 and x>=0");
                if(x>=a+1) throw new IllegalArgumentException("\nx >= a+1   use Continued Fraction Representation");

                int i = 0;
                double igf = 0.0D;
                boolean check = true;

                double acopy = a;
                double sum = 1.0/a;
                double incr = sum;
                double loggamma = Stat.logGamma(a);

                while(check){
                        ++i;
                        ++a;
                        incr *= x/a;
                        sum += incr;
                        if(Math.abs(incr) < Math.abs(sum)*Stat.igfeps){
                                igf = sum*Math.exp(-x+acopy*Math.log(x)- loggamma);
                                check = false;
                        }
                        if(i>=Stat.igfiter){
                                check=false;
                                igf = sum*Math.exp(-x+acopy*Math.log(x)- loggamma);
                                System.out.println("\nMaximum number of iterations were exceeded in Stat.incompleteGammaSer().\nCurrent value returned.\nIncrement = "+String.valueOf(incr)+".\nSum = "+String.valueOf(sum)+".\nTolerance =  "+String.valueOf(igfeps));
                        }
                }
                return igf;
        }

        // Regularised Incomplete Gamma Function P(a,x) = integral from zero to x of (exp(-t)t^(a-1))dt
        // Continued Fraction representation of the function - valid for x >= a + 1
        // This method follows the general procedure used in Numerical Recipes for C,
        // The Art of Scientific Computing
        // by W H Press, S A Teukolsky, W T Vetterling & B P Flannery
        // Cambridge University Press,   http://www.nr.com/
        public static double incompleteGammaFract(double a, double x){
                if(a<0.0D  || x<0.0D)throw new IllegalArgumentException("\nFunction defined only for a >= 0 and x>=0");
                if(x<a+1) throw new IllegalArgumentException("\nx < a+1   Use Series Representation");

                int i = 0;
                double ii = 0;
                double igf = 0.0D;
                boolean check = true;

                double loggamma = Stat.logGamma(a);
                double numer = 0.0D;
                double incr = 0.0D;
                double denom = x - a + 1.0D;
                double first = 1.0D/denom;
                double term = 1.0D/FPMIN;
                double prod = first;

                while(check){
                        ++i;
                        ii = (double)i;
                        numer = -ii*(ii - a);
                        denom += 2.0D;
                        first = numer*first + denom;
                        if(Math.abs(first) < Stat.FPMIN){
                            first = Stat.FPMIN;
                        }
                        term = denom + numer/term;
                        if(Math.abs(term) < Stat.FPMIN){
                            term = Stat.FPMIN;
                         }
                        first = 1.0D/first;
                        incr = first*term;
                        prod *= incr;
                        if(Math.abs(incr - 1.0D) < igfeps)check = false;
                        if(i>=Stat.igfiter){
                                check=false;
                                System.out.println("\nMaximum number of iterations were exceeded in Stat.incompleteGammaFract().\nCurrent value returned.\nIncrement - 1 = "+String.valueOf(incr-1)+".\nTolerance =  "+String.valueOf(igfeps));
                        }
                }
                igf = 1.0D - Math.exp(-x+a*Math.log(x)-loggamma)*prod;
                return igf;
        }

        // Reset the maximum number of iterations allowed in the calculation of the incomplete gamma functions
        public static void setIncGammaMaxIter(int igfiter){
                Stat.igfiter=igfiter;
        }

        // Return the maximum number of iterations allowed in the calculation of the incomplete gamma functions
        public static int getIncGammaMaxIter(){
                return Stat.igfiter;
        }

        // Reset the tolerance used in the calculation of the incomplete gamma functions
        public static void setIncGammaTol(double igfeps){
                Stat.igfeps=igfeps;
        }

        // Return the tolerance used in the calculation of the incomplete gamm functions
        public static double getIncGammaTol(){
                return Stat.igfeps;
        }

        // FACTORIALS

        // factorial of n
        // argument and return are integer, therefore limited to 0<=n<=12
        // see below for long and double arguments
        public static int factorial(int n){
            if(n<0)throw new IllegalArgumentException("n must be a positive integer");
            if(n>12)throw new IllegalArgumentException("n must less than 13 to avoid integer overflow\nTry long or double argument");
            int f = 1;
            for(int i=2; i<=n; i++)f*=i;
            return f;
        }

        // factorial of n
        // argument and return are long, therefore limited to 0<=n<=20
        // see below for double argument
        public static long factorial(long n){
            if(n<0)throw new IllegalArgumentException("n must be a positive integer");
            if(n>20)throw new IllegalArgumentException("n must less than 21 to avoid long integer overflow\nTry double argument");
            long f = 1;
            long iCount = 2L;
            while(iCount<=n){
                f*=iCount;
                iCount += 1L;
            }
            return f;
        }

        // factorial of n
        // Argument is of type BigInteger
        public static BigInteger factorial(BigInteger n){
            if(n.compareTo(BigInteger.ZERO)==-1)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            BigInteger one = BigInteger.ONE;
            BigInteger f = one;
            BigInteger iCount = new BigInteger("2");
            while(iCount.compareTo(n)!=1){
                f = f.multiply(iCount);
                iCount = iCount.add(one);
            }
            one = null;
            iCount = null;
            return f;
        }

        // factorial of n
        // Argument is of type double but must be, numerically, an integer
        // factorial returned as double but is, numerically, should be an integer
        // numerical rounding may makes this an approximation after n = 21
        public static double factorial(double n){
            if(n<0 || (n-Math.floor(n))!=0)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            double f = 1.0D;
            double iCount = 2.0D;
            while(iCount<=n){
                f*=iCount;
                iCount += 1.0D;
            }
            return f;
        }

        // factorial of n
        // Argument is of type BigDecimal but must be, numerically, an integer
        public static BigDecimal factorial(BigDecimal n){
            if(n.compareTo(BigDecimal.ZERO)==-1 || !Fmath.isInteger(n))throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            BigDecimal one = BigDecimal.ONE;
            BigDecimal f = one;
            BigDecimal iCount = new BigDecimal(2.0D);
            while(iCount.compareTo(n)!=1){
                f = f.multiply(iCount);
                iCount = iCount.add(one);
            }
            one = null;
            iCount = null;
            return f;
        }


        // log to base e of the factorial of n
        // log[e](factorial) returned as double
        // numerical rounding may makes this an approximation
        public static double logFactorial(int n){
            if(n<0)throw new IllegalArgumentException("\nn, " + n + ", must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            double f = 0.0D;
            for(int i=2; i<=n; i++)f+=Math.log(i);
            return f;
        }

        // log to base e of the factorial of n
        // Argument is of type double but must be, numerically, an integer
        // log[e](factorial) returned as double
        // numerical rounding may makes this an approximation
        public static double logFactorial(long n){
            if(n<0)throw new IllegalArgumentException("\nn, " + n + ", must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            double f = 0.0D;
            long iCount = 2L;
            while(iCount<=n){
                f+=Math.log(iCount);
                iCount += 1L;
            }
            return f;
        }

        // log to base e of the factorial of n
        // Argument is of type double but must be, numerically, an integer
        // log[e](factorial) returned as double
        // numerical rounding may makes this an approximation
        public static double logFactorial(double n){
            if(n<0 || (n-Math.floor(n))!=0)throw new IllegalArgumentException("\nn must be a positive integer\nIs a Gamma funtion [Fmath.gamma(x)] more appropriate?");
            double f = 0.0D;
            double iCount = 2.0D;
            while(iCount<=n){
                f+=Math.log(iCount);
                iCount += 1.0D;
            }
            return f;
        }


        // ERLANG DISTRIBUTION AND ERLANG EQUATIONS

        // Erlang distribution
        // cumulative distribution function
        public static double erlangCDF(double lambda, int kay, double upperLimit){
            return gammaCDF(0.0D, 1.0D/lambda, (double)kay, upperLimit);
        }

        public static double erlangCDF(double lambda, long kay, double upperLimit){
            return gammaCDF(0.0D, 1.0D/lambda, (double)kay, upperLimit);
        }

        public static double erlangCDF(double lambda, double kay, double upperLimit){
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            return gammaCDF(0.0D, 1.0D/lambda, kay, upperLimit);
        }

        // Erlang distribution
        // probablity density function
        public static double erlangPDF(double lambda, int kay, double x){
            return  gammaPDF(0.0D, 1.0D/lambda, (double)kay, x);
        }

        public static double erlangPDF(double lambda, long kay, double x){
            return  gammaPDF(0.0D, 1.0D/lambda, (double)kay, x);
        }

        public static double erlangPDF(double lambda, double kay, double x){
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");

            return  gammaPDF(0.0D, 1.0D/lambda, kay, x);
        }

        // Erlang distribution
        // mean
        public static double erlangMean(double lambda, int kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return (double)kay/lambda;
        }

        public static double erlangMean(double lambda, long kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return (double)kay/lambda;
        }

        public static double erlangMean(double lambda, double kay){
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return kay/lambda;
        }

        // erlang distribution
        // mode
        public static double erlangMode(double lambda, int kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            double mode = Double.NaN;
            if(kay>=1)mode = ((double)kay-1.0D)/lambda;
            return mode;
        }

        public static double erlangMode(double lambda, long kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            double mode = Double.NaN;
            if(kay>=1)mode = ((double)kay-1.0D)/lambda;
            return mode;
        }

        public static double erlangMode(double lambda, double kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            double mode = Double.NaN;
            if(kay>=1)mode = (kay-1.0D)/lambda;
            return mode;
        }


        // Erlang distribution
        // standard deviation
        public static double erlangStandDev(double lambda, int kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return Math.sqrt((double)kay)/lambda;
        }

        public static double erlangStandDev(double lambda, long kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return Math.sqrt((double)kay)/lambda;
        }

        public static double erlangStandDev(double lambda, double kay){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            return Math.sqrt(kay)/lambda;
        }

        // Returns an array of Erlang random deviates - clock seed
        public static double[] erlangRand(double lambda, int kay, int n){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return gammaRand(0.0D, 1.0D/lambda, (double) kay, n);
        }

        public static double[] erlangRand(double lambda, long kay, int n){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return gammaRand(0.0D, 1.0D/lambda, (double) kay, n);
        }

        public static double[] erlangRand(double lambda, double kay, int n){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            return gammaRand(0.0D, 1.0D/lambda, kay, n);
        }

       // Returns an array of Erlang random deviates - user supplied seed
        public static double[] erlangRand(double lambda, int kay, int n, long seed){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return gammaRand(0.0D, 1.0D/lambda, (double) kay, n, seed);
        }

        public static double[] erlangRand(double lambda, long kay, int n, long seed){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            return gammaRand(0.0D, 1.0D/lambda, (double) kay, n, seed);
        }

        public static double[] erlangRand(double lambda, double kay, int n, long seed){
            if(kay<1)throw new IllegalArgumentException("The rate parameter, " + kay + "must be equal to or greater than one");
            if(kay - Math.round(kay)!=0.0D)throw new IllegalArgumentException("kay must, mathematically, be an integer even though it may be entered as a double\nTry the Gamma distribution instead of the Erlang distribution");
            return gammaRand(0.0D, 1.0D/lambda, kay, n, seed);
        }


        // ERLANG CONNECTIONS BUSY, B AND C EQUATIONS

        // returns the probablility that m resources (connections) are busy
        // totalTraffic:    total traffic in Erlangs
        // totalResouces:   total number of resources in the system
        public static double erlangMprobability(double totalTraffic, double totalResources, double em){
            double prob = 0.0D;
            if(totalTraffic>0.0D){

                double numer = totalResources*Math.log(em) - Fmath.logFactorial(em);
                double denom = 1.0D;
                double lastTerm = 1.0D;
                for(int i=1; i<=totalResources; i++){
                    lastTerm = lastTerm*totalTraffic/(double)i;
                    denom += lastTerm;
                }
                denom = Math.log(denom);
                prob = numer - denom;
                prob = Math.exp(prob);
            }
            return prob;
        }

        public static double erlangMprobability(double totalTraffic, long totalResources, long em){
            return erlangMprobability(totalTraffic, (double)totalResources, (double)em);
        }

        public static double erlangMprobability(double totalTraffic, int totalResources, int em){
            return erlangMprobability(totalTraffic, (double)totalResources, (double)em);
        }


        // Erlang B equation
        // returns the probablility that a customer will be rejected due to lack of resources
        // totalTraffic:    total traffic in Erlangs
        // totalResouces:   total number of resources in the system
        public static double erlangBprobability(double totalTraffic, double totalResources){
            double prob = 0.0D;
            if(totalTraffic>0.0D){
                double numer = totalResources*Math.log(totalTraffic) - Fmath.logFactorial(totalResources);
                double denom = 1.0D;
                double lastTerm = 1.0D;
                double iCount = 1.0D;
                while(iCount<=totalResources){
                    lastTerm = lastTerm*totalTraffic/iCount;
                    denom += lastTerm;
                    iCount = iCount + 1.0D;
                }
                denom = Math.log(denom);
                prob = numer - denom;
                prob = Math.exp(prob);
            }
            return prob;
        }

        public static double erlangBprobability(double totalTraffic, long totalResources){
            return erlangBprobability(totalTraffic, (double)totalResources);
        }

        public static double erlangBprobability(double totalTraffic, int totalResources){
            return erlangBprobability(totalTraffic, (double)totalResources);
        }

        // Erlang B equation
        // returns the maximum total traffic in Erlangs
        // blockingProbability:    probablility that a customer will be rejected due to lack of resources
        // totalResouces:   total number of resources in the system
        public static double erlangBload(double blockingProbability, double totalResources){

            // Create instance of the class holding the Erlang B equation
            ErlangBfunct eBfunc = new ErlangBfunct();

            // Set instance variables
            eBfunc.blockingProbability = blockingProbability;
            eBfunc.totalResources = totalResources;

            // lower bound
            double lowerBound = 0.0D;
            // upper bound
            double upperBound = totalResources*0.999999D;
            // required tolerance
            double tolerance = 1e-6;

            // Create instance of RealRoot
            RealRoot realR = new RealRoot();

            // Set tolerance
            realR.setTolerance(tolerance);

            // Set bounds limits
            realR.noLowerBoundExtension();

            // Supress error message if iteration limit reached
            realR.supressLimitReachedMessage();

            // call root searching method
            double root = realR.bisect(eBfunc, lowerBound, upperBound);

            return root;
        }

        public static double erlangBload(double blockingProbability, long totalResources){
            return erlangBload(blockingProbability, (double)totalResources);
        }

        public static double erlangBload(double blockingProbability, int totalResources){
            return erlangBload(blockingProbability, (double)totalResources);
        }

        // Erlang C equation
        // returns the probablility that a customer will receive a non-zero delay in obtaining obtaining a resource
        // totalTraffic:    total traffic in Erlangs
        // totalResouces:   total number of resources in the system
        public static double erlangCprobability(double totalTraffic, double totalResources){
            double prob = 0.0D;
            if(totalTraffic>0.0D){

                double numer = totalResources*Math.log(totalTraffic) - Fmath.logFactorial(totalResources);
                numer = Math.exp(numer)*totalResources/(totalResources - totalTraffic);
                double denom = 1.0D;
                double lastTerm = 1.0D;
                double iCount = 1.0D;
                while(iCount<=totalResources){
                    lastTerm = lastTerm*totalTraffic/iCount;
                    denom += lastTerm;
                    iCount = iCount + 1.0D;
                }
                denom += numer;
                prob = numer/denom;
            }
            return prob;
        }

        public static double erlangCprobability(double totalTraffic, long totalResources){
            return erlangCprobability(totalTraffic, (double)totalResources);
        }

        public static double erlangCprobability(double totalTraffic, int totalResources){
            return erlangCprobability(totalTraffic, (double)totalResources);
        }

        // Erlang C equation
        // returns the maximum total traffic in Erlangs
        // nonZeroDelayProbability:    probablility that a customer will receive a non-zero delay in obtaining obtaining a resource
        // totalResouces:   total number of resources in the system
        public static double erlangCload(double nonZeroDelayProbability, double totalResources){

            // Create instance of the class holding the Erlang C equation
            ErlangCfunct eCfunc = new ErlangCfunct();

            // Set instance variables
            eCfunc.nonZeroDelayProbability = nonZeroDelayProbability;
            eCfunc.totalResources = totalResources;

            // lower bound
            double lowerBound = 0.0D;
            // upper bound
            double upperBound = 1.0D;
            // required tolerance
            double tolerance = 1e-6;

            // Create instance of RealRoot
            RealRoot realR = new RealRoot();

            // Set tolerance
            realR.setTolerance(tolerance);

            // Supress error message if iteration limit reached
            realR.supressLimitReachedMessage();

            // Set bounds limits
            realR.noLowerBoundExtension();

            // call root searching method
            double root = realR.bisect(eCfunc, lowerBound, upperBound);

            return root;
        }

        public static double erlangCload(double nonZeroDelayProbability, long totalResources){
            return erlangCload(nonZeroDelayProbability, (double)totalResources);
        }

        public static double erlangCload(double nonZeroDelayProbability, int totalResources){
            return erlangCload(nonZeroDelayProbability, (double)totalResources);
        }


        // ENGSET EQUATION

        // returns the probablility that a customer will be rejected due to lack of resources
        // offeredTraffic:  total offeredtraffic in Erlangs
        // totalResouces:   total number of resources in the system
        // numberOfSources: number of sources
        public static double engsetProbability(double offeredTraffic, double totalResources, double numberOfSources){
            if(totalResources>numberOfSources-1)throw new IllegalArgumentException("total resources, " + totalResources + ", must be less than or  equal to the number of sources minus one, " + (numberOfSources - 1));

            double prob = 0.0D;
            if(totalResources==0.0D){
                prob = 1.0D;
            }
            else{
                if(offeredTraffic==0.0D){
                    prob = 0.0D;
                }
                else{
                    // Set boundaries to the probability
                    double lowerBound = 0.0D;
                    double upperBound = 1.0D;

                    // Create instance of Engset Probability Function
                    EngsetProb engProb = new EngsetProb();

                    // Set function variables
                    engProb.offeredTraffic = offeredTraffic;
                    engProb.totalResources = totalResources;
                    engProb.numberOfSources = numberOfSources;

                    // Perform a root search
                    RealRoot eprt = new RealRoot();

                    // Supress error message if iteration limit reached
                    eprt.supressLimitReachedMessage();

                    prob = eprt.bisect(engProb, lowerBound, upperBound);
                }
            }
            return prob;
        }

        public static double engsetProbability(double offeredTraffic, long totalResources, long numberOfSources){
            return engsetProbability(offeredTraffic, (double)totalResources, (double)numberOfSources);
        }

        public static double engsetProbability(double offeredTraffic, int totalResources, int numberOfSources){
            return engsetProbability(offeredTraffic, (double)totalResources, (double)numberOfSources);
        }

        // Engset equation
        // returns the maximum total traffic in Erlangs
        // blockingProbability:    probablility that a customer will be rejected due to lack of resources
        // totalResouces:   total number of resources in the system
        // numberOfSources: number of sources
        public static double engsetLoad(double blockingProbability, double totalResources, double numberOfSources){

            // Create instance of the class holding the Engset Load equation
            EngsetLoad eLfunc = new EngsetLoad();

            // Set instance variables
            eLfunc.blockingProbability = blockingProbability;
            eLfunc.totalResources = totalResources;
            eLfunc.numberOfSources = numberOfSources;

            // lower bound
            double lowerBound = 0.0D;
            // upper bound
            double upperBound = totalResources*0.999999D;
            // required tolerance
            double tolerance = 1e-6;

            // Create instance of RealRoot
            RealRoot realR = new RealRoot();

            // Set tolerance
            realR.setTolerance(tolerance);

            // Set bounds limits
            realR.noLowerBoundExtension();

            // Supress error message if iteration limit reached
            realR.supressLimitReachedMessage();

            // call root searching method
            double root = realR.bisect(eLfunc, lowerBound, upperBound);

            return root;
        }

        public static double engsetLoad(double blockingProbability, long totalResources, long numberOfSources){
            return engsetLoad(blockingProbability, (double) totalResources, (double) numberOfSources);
        }

        public static double engsetLoad(double blockingProbability, int totalResources, int numberOfSources){
            return engsetLoad(blockingProbability, (double) totalResources, (double) numberOfSources);
        }

        // BETA DISTRIBUTIONS AND BETA FUNCTIONS

        // beta distribution cdf
        public static double betaCDF(double alpha, double beta, double limit){
            return betaCDF(0.0D, 1.0D, alpha, beta, limit);
        }

        // beta distribution pdf
        public static double betaCDF(double min, double max, double alpha, double beta, double limit){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            if(limit<min)throw new IllegalArgumentException("limit, " + limit + ", must be greater than or equal to the minimum value, " + min);
            if(limit>max)throw new IllegalArgumentException("limit, " + limit + ", must be less than or equal to the maximum value, " + max);
            return Stat.regularisedBetaFunction(alpha, beta, (limit-min)/(max-min));
        }


        // beta distribution pdf
        public static double betaPDF(double alpha, double beta, double x){
            return betaPDF(0.0D, 1.0D, alpha, beta, x);
        }

        // beta distribution pdf
        public static double betaPDF(double min, double max, double alpha, double beta, double x){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            if(x<min)throw new IllegalArgumentException("x, " + x + ", must be greater than or equal to the minimum value, " + min);
            if(x>max)throw new IllegalArgumentException("x, " + x + ", must be less than or equal to the maximum value, " + max);
            double pdf = Math.pow(x - min, alpha - 1)*Math.pow(max - x, beta - 1)/Math.pow(max - min, alpha + beta - 1);
            return pdf/Stat.betaFunction(alpha, beta);
        }

        // Returns an array of Beta random deviates - clock seed
        public static double[] betaRand(double alpha, double beta, int n){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            PsRandom psr = new PsRandom();
            return psr.betaArray(alpha, beta, n);
        }

        // Returns an array of Beta random deviates - clock seed
        public static double[] betaRand(double min, double max, double alpha, double beta, int n){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            PsRandom psr = new PsRandom();
            return psr.betaArray(min, max, alpha, beta, n);
        }


        // Returns an array of Beta random deviates - user supplied seed
        public static double[] betaRand(double alpha, double beta, int n, long seed){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter,  beta, " + beta + "must be greater than zero");
            PsRandom psr = new PsRandom(seed);
            return psr.betaArray(alpha, beta, n);
        }

        // Returns an array of Beta random deviates - user supplied seed
        public static double[] betaRand(double min, double max, double alpha, double beta, int n, long seed){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter,  beta, " + beta + "must be greater than zero");
            PsRandom psr = new PsRandom(seed);
            return psr.betaArray(min, max, alpha, beta, n);
        }

        // beta distribution mean
        public static double betaMean(double alpha, double beta){
            return betaMean(0.0D, 1.0D, alpha, beta);
        }

        // beta distribution mean
        public static double betaMean(double min, double max, double alpha, double beta){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            return min + alpha*(max - min)/(alpha + beta);
        }

        // beta distribution mode
        public static double betaMode(double alpha, double beta){
            return betaMode(0.0D, 1.0D, alpha, beta);
        }

        // beta distribution mode
        public static double betaMode(double min, double max, double alpha, double beta){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");

            double mode = Double.NaN;
            if(alpha>1){
                if(beta>1){
                    mode = min + (alpha + beta)*(max - min)/(alpha + beta - 2);
                }
                else{
                    mode = max;
                }
            }
            else{
                if(alpha==1){
                    if(beta>1){
                        mode = min;
                    }
                    else{
                        if(beta==1){
                            mode = Double.NaN;
                        }
                        else{
                            mode = max;
                        }
                    }
                }
                else{
                    if(beta>=1){
                        mode = min;
                    }
                    else{
                        System.out.println("Class Stat; method betaMode; distribution is bimodal wirh modes at " + min + " and " + max);
                        System.out.println("NaN returned");
                    }
                }
            }
            return mode;
        }

        // beta distribution standard deviation
        public static double betaStandDev(double alpha, double beta){
            return betaStandDev(0.0D, 1.0D, alpha, beta);
        }

        // beta distribution standard deviation
        public static double betaStandDev(double min, double max, double alpha, double beta){
            if(alpha<=0.0D)throw new IllegalArgumentException("The shape parameter, alpha, " + alpha + "must be greater than zero");
            if(beta<=0.0D)throw new IllegalArgumentException("The shape parameter, beta, " + beta + "must be greater than zero");
            return ((max - min)/(alpha + beta))*Math.sqrt(alpha*beta/(alpha + beta + 1));
        }

        // Beta function
        public static double betaFunction(double z, double w){
                return Math.exp(logGamma(z) + logGamma(w) - logGamma(z + w));
        }

        // Beta function
        // retained for compatibility reasons
        public static double beta(double z, double w){
                return Math.exp(logGamma(z) + logGamma(w) - logGamma(z + w));
        }

        // Regularised Incomplete Beta function
        // Continued Fraction approximation (see Numerical recipies for details of method)
        public static double regularisedBetaFunction(double z, double w, double x){
            if(x<0.0D || x>1.0D)throw new IllegalArgumentException("Argument x, "+x+", must be lie between 0 and 1 (inclusive)");
            double ibeta = 0.0D;
            if(x==0.0D){
                ibeta=0.0D;
            }
            else{
                if(x==1.0D){
                    ibeta=1.0D;
                }
                else{
                    // Term before continued fraction
                    ibeta = Math.exp(Stat.logGamma(z+w) - Stat.logGamma(z) - logGamma(w) + z*Math.log(x) + w*Math.log(1.0D-x));
                    // Continued fraction
                    if(x < (z+1.0D)/(z+w+2.0D)){
                        ibeta = ibeta*Stat.contFract(z, w, x)/z;
                    }
                    else{
                        // Use symmetry relationship
                        ibeta = 1.0D - ibeta*Stat.contFract(w, z, 1.0D-x)/w;
                    }
                }
            }
            return ibeta;
        }


        // Regularised Incomplete Beta function
        // Continued Fraction approximation (see Numerical recipies for details of method)
        public static double regularizedBetaFunction(double z, double w, double x){
            return regularisedBetaFunction(z, w, x);
        }

        // Regularised Incomplete Beta function
        // Continued Fraction approximation (see Numerical recipies for details of method)
        // retained for compatibility reasons
        public static double incompleteBeta(double z, double w, double x){
            return regularisedBetaFunction(z, w, x);
        }

        // Incomplete fraction summation used in the method regularisedBetaFunction
        // modified Lentz's method
        public static double contFract(double a, double b, double x){
            int maxit = 500;
            double eps = 3.0e-7;
            double aplusb = a + b;
            double aplus1 = a + 1.0D;
            double aminus1 = a - 1.0D;
            double c = 1.0D;
            double d = 1.0D - aplusb*x/aplus1;
            if(Math.abs(d)<Stat.FPMIN)d = FPMIN;
            d = 1.0D/d;
            double h = d;
            double aa = 0.0D;
            double del = 0.0D;
            int i=1, i2=0;
            boolean test=true;
            while(test){
                i2=2*i;
                aa = i*(b-i)*x/((aminus1+i2)*(a+i2));
                d = 1.0D + aa*d;
                if(Math.abs(d)<Stat.FPMIN)d = FPMIN;
                c = 1.0D + aa/c;
                if(Math.abs(c)<Stat.FPMIN)c = FPMIN;
                d = 1.0D/d;
                h *= d*c;
                aa = -(a+i)*(aplusb+i)*x/((a+i2)*(aplus1+i2));
                d = 1.0D + aa*d;
                if(Math.abs(d)<Stat.FPMIN)d = FPMIN;
                c = 1.0D + aa/c;
                if(Math.abs(c)<Stat.FPMIN)c = FPMIN;
                d = 1.0D/d;
                del = d*c;
                h *= del;
                i++;
                if(Math.abs(del-1.0D) < eps)test=false;
                if(i>maxit){
                    test=false;
                    System.out.println("Maximum number of iterations ("+maxit+") exceeded in Stat.contFract in Stat.incomplete Beta");
                }
            }
            return h;

        }


        // ERROR FUNCTIONS

        // Error Function
        public static double erf(double x){
                double erf = 0.0D;
                if(x!=0.0){
                        if(x==1.0D/0.0D){
                                erf = 1.0D;
                        }
                        else{
                                if(x>=0){
                                        erf = Stat.incompleteGamma(0.5, x*x);
                                }
                                else{
                                        erf = -Stat.incompleteGamma(0.5, x*x);
                                }
                        }
                }
                return erf;
        }

        // Complementary Error Function
        public static double erfc(double x){
                double erfc = 1.0D;
                if(x!=0.0){
                        if(x==1.0D/0.0D){
                                erfc = 0.0D;
                        }
                        else{
                                if(x>=0){
                                        erfc = 1.0D - Stat.incompleteGamma(0.5, x*x);
                                }
                                else{
                                        erfc = 1.0D + Stat.incompleteGamma(0.5, x*x);
                                }
                        }
                }
                return erfc;
        }


        // NORMAL (GAUSSIAN) DISTRIBUTION

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mean  =  the mean, sd = standard deviation
        public static double normalCDF(double mean, double sd, double upperlimit){
                double arg = (upperlimit - mean)/(sd*Math.sqrt(2.0));
                return (1.0D + Stat.erf(arg))/2.0D;
        }
        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mean  =  the mean, sd = standard deviation
        public static double normalProb(double mean, double sd, double upperlimit){
                double arg = (upperlimit - mean)/(sd*Math.sqrt(2.0));
                return (1.0D + Stat.erf(arg))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mean  =  the mean, sd = standard deviation
        public static double gaussianCDF(double mean, double sd, double upperlimit){
                double arg = (upperlimit - mean)/(sd*Math.sqrt(2.0));
                return (1.0D + Stat.erf(arg))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mean  =  the mean, sd = standard deviation
        public static double gaussianProb(double mean, double sd, double upperlimit){
                double arg = (upperlimit - mean)/(sd*Math.sqrt(2.0));
                return (1.0D + Stat.erf(arg))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mean  =  the mean, sd = standard deviation
        public static double normalCDF(double mean, double sd, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mean)/(sd*Math.sqrt(2.0));
                double arg2 = (upperlimit - mean)/(sd*Math.sqrt(2.0));

                return (Stat.erf(arg2)-Stat.erf(arg1))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mean  =  the mean, sd = standard deviation
        public static double normalProb(double mean, double sd, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mean)/(sd*Math.sqrt(2.0));
                double arg2 = (upperlimit - mean)/(sd*Math.sqrt(2.0));

                return (Stat.erf(arg2)-Stat.erf(arg1))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mean  =  the mean, sd = standard deviation
        public static double gaussianCDF(double mean, double sd, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mean)/(sd*Math.sqrt(2.0));
                double arg2 = (upperlimit - mean)/(sd*Math.sqrt(2.0));

                return (Stat.erf(arg2)-Stat.erf(arg1))/2.0D;
        }

        // Gaussian (normal) cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mean  =  the mean, sd = standard deviation
        public static double gaussianProb(double mean, double sd, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mean)/(sd*Math.sqrt(2.0));
                double arg2 = (upperlimit - mean)/(sd*Math.sqrt(2.0));

                return (Stat.erf(arg2)-Stat.erf(arg1))/2.0D;
        }

        // Gaussian (normal) probability
        // mean  =  the mean, sd = standard deviation
        public static double normalPDF(double mean, double sd, double x){
                return Math.exp(-Fmath.square((x - mean)/sd)/2.0)/(sd*Math.sqrt(2.0D*Math.PI));
        }

        // Gaussian (normal) probability
        // mean  =  the mean, sd = standard deviation
        public static double normal(double mean, double sd, double x){
                return Math.exp(-Fmath.square((x - mean)/sd)/2.0)/(sd*Math.sqrt(2.0D*Math.PI));
        }

         // Gaussian (normal) probability
        // mean  =  the mean, sd = standard deviation
        public static double gaussianPDF(double mean, double sd, double x){
                return Math.exp(-Fmath.square((x - mean)/sd)/2.0)/(sd*Math.sqrt(2.0D*Math.PI));
        }
        // Gaussian (normal) probability
        // mean  =  the mean, sd = standard deviation
        public static double gaussian(double mean, double sd, double x){
                return Math.exp(-Fmath.square((x - mean)/sd)/2.0)/(sd*Math.sqrt(2.0D*Math.PI));
        }

        // Returns an array of Gaussian (normal) random deviates - clock seed
        // mean  =  the mean, sd = standard deviation, length of array
        public static double[] normalRand(double mean, double sd, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                    ran[i]=rr.nextGaussian();
                    ran[i] = ran[i]*sd+mean;
                }
                return ran;
        }

        // Returns an array of Gaussian (normal) random deviates - clock seed
        // mean  =  the mean, sd = standard deviation, length of array
        public static double[] gaussianRand(double mean, double sd, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                    ran[i]=rr.nextGaussian();
                    ran[i] = ran[i]*sd+mean;
                }
                return ran;
        }

        // Returns an array of Gaussian (normal) random deviates - user provided seed
        // mean  =  the mean, sd = standard deviation, length of array
        public static double[] normalRand(double mean, double sd, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                    ran[i]=rr.nextGaussian();
                    ran[i] = ran[i]*sd+mean;
                }
                return ran;
        }

        // Returns an array of Gaussian (normal) random deviates - user provided seed
        // mean  =  the mean, sd = standard deviation, length of array
        public static double[] gaussianRand(double mean, double sd, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                    ran[i]=rr.nextGaussian();
                    ran[i] = ran[i]*sd+mean;
                }
                return ran;
        }


        // LOG-NORMAL DISTRIBUTIONS (TWO AND THEE PARAMETER DISTRIBUTIONS)

        // TWO PARAMETER LOG-NORMAL DISTRIBUTION

        // Two parameter log-normal cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double logNormalCDF(double mu, double sigma, double upperLimit){
            if(sigma<0)throw new IllegalArgumentException("The parameter sigma, " + sigma + ", must be greater than or equal to zero");
            if(upperLimit<=0){
                    return 0.0D;
                }
                else{
                    return 0.5D*(1.0D + Stat.erf((Math.log(upperLimit)-mu)/(sigma*Math.sqrt(2))));
                }
        }

        public static double logNormalTwoParCDF(double mu, double sigma, double upperLimit){
            return logNormalCDF(mu, sigma, upperLimit);
        }


        // Two parameter log-normal cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double logNormalCDF(double mu, double sigma, double lowerLimit, double upperLimit){
                if(sigma<0)throw new IllegalArgumentException("The parameter sigma, " + sigma + ", must be greater than or equal to zero");
                if(upperLimit<lowerLimit)throw new IllegalArgumentException("The upper limit, " + upperLimit + ", must be greater than the " + lowerLimit);

                double arg1 = 0.0D;
                double arg2 = 0.0D;
                double cdf = 0.0D;

                if(lowerLimit!=upperLimit){
                    if(upperLimit>0.0D)arg1 = 0.5D*(1.0D + Stat.erf((Math.log(upperLimit)-mu)/(sigma*Math.sqrt(2))));
                    if(lowerLimit>0.0D)arg2 = 0.5D*(1.0D + Stat.erf((Math.log(lowerLimit)-mu)/(sigma*Math.sqrt(2))));
                    cdf = arg1 - arg2;
                }

                return cdf;
        }

        public static double logNormalTwoParCDF(double mu, double sigma, double lowerLimit, double upperLimit){
            return logNormalCDF(mu, sigma, lowerLimit, upperLimit);
        }


        // Two parameter log-normal probability
        public static double logNormalPDF(double mu, double sigma, double x){
            if(sigma<0)throw new IllegalArgumentException("The parameter sigma, " + sigma + ", must be greater than or equal to zero");
            if(x<0){
                return 0.0D;
            }
            else{
                return Math.exp(-0.5D*Fmath.square((Math.log(x)- mu)/sigma))/(x*sigma*Math.sqrt(2.0D*Math.PI));
            }
        }

        public static double logNormalTwoParPDF(double mu, double sigma, double x){
            return logNormalPDF(mu, sigma, x);
        }


        // Two parameter log-normal mean
        public static double logNormalMean(double mu, double sigma){
                return Math.exp(mu + sigma*sigma/2.0D);
        }

        public static double logNormalTwoParMean(double mu, double sigma){
                return Math.exp(mu + sigma*sigma/2.0D);
        }


        // Two parameter log-normal standard deviation
        public static double logNormalStandDev(double mu, double sigma){
                double sigma2 = sigma*sigma;
                return Math.sqrt((Math.exp(sigma2) - 1.0D)*Math.exp(2.0D*mu + sigma2));
        }

        public static double logNormalTwoParStandDev(double mu, double sigma){
                double sigma2 = sigma*sigma;
                return Math.sqrt((Math.exp(sigma2) - 1.0D)*Math.exp(2.0D*mu + sigma2));
        }

        // Two parameter log-normal mode
        public static double logNormalMode(double mu, double sigma){
            return Math.exp(mu - sigma*sigma);
        }

        public static double logNormalTwoParMode(double mu, double sigma){
            return Math.exp(mu - sigma*sigma);
        }

        // Two parameter log-normal median
        public static double logNormalMedian(double mu){
            return Math.exp(mu);
        }

        public static double logNormalTwoParMedian(double mu){
            return Math.exp(mu);
        }

        // Returns an array of two parameter log-normal random deviates - clock seed
        public static double[] logNormalRand(double mu, double sigma, int n){
            if(n<=0)throw new IllegalArgumentException("The number of random deviates required, " + n + ", must be greater than zero");
            if(sigma<0)throw new IllegalArgumentException("The parameter sigma, " + sigma + ", must be greater than or equal to zero");
            PsRandom psr = new PsRandom();
            return psr.logNormalArray(mu, sigma, n);
        }

        public static double[] logNormalTwoParRand(double mu, double sigma, int n){
            return logNormalRand(mu, sigma, n);
        }


        // Returns an array of two parameter log-normal random deviates - user supplied seed
        public static double[] logNormalRand(double mu, double sigma, int n, long seed){
            if(n<=0)throw new IllegalArgumentException("The number of random deviates required, " + n + ", must be greater than zero");
            if(sigma<0)throw new IllegalArgumentException("The parameter sigma, " + sigma + ", must be greater than or equal to zero");
            PsRandom psr = new PsRandom(seed);
            return psr.logNormalArray(mu, sigma, n);
        }

        public static double[] logNormalTwoParRand(double mu, double sigma, int n, long seed){
            return logNormalRand(mu, sigma, n, seed);
        }

        // THREE PARAMETER LOG-NORMAL DISTRIBUTION

        // Three parameter log-normal cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double logNormalThreeParCDF(double alpha, double beta, double gamma,  double upperLimit){
            if(beta<0)throw new IllegalArgumentException("The parameter beta, " + beta + ", must be greater than or equal to zero");
            if(upperLimit<=0){
                    return 0.0D;
                }
                else{
                    return 0.5D*(1.0D + Stat.erf(Math.log((upperLimit-alpha)/gamma)/(beta*Math.sqrt(2))));
                }
        }


        // Three parameter log-normal cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double logNormalThreeParCDF(double alpha, double beta, double gamma,  double lowerLimit, double upperLimit){
                if(beta<0)throw new IllegalArgumentException("The parameter beta, " + beta + ", must be greater than or equal to zero");
                if(upperLimit<lowerLimit)throw new IllegalArgumentException("The upper limit, " + upperLimit + ", must be greater than the " + lowerLimit);

                double arg1 = 0.0D;
                double arg2 = 0.0D;
                double cdf = 0.0D;

                if(lowerLimit!=upperLimit){
                    if(upperLimit>0.0D)arg1 = 0.5D*(1.0D + Stat.erf(Math.log((upperLimit-alpha)/gamma)/(beta*Math.sqrt(2))));
                    if(lowerLimit>0.0D)arg2 = 0.5D*(1.0D + Stat.erf(Math.log((lowerLimit-alpha)/gamma)/(beta*Math.sqrt(2))));
                    cdf = arg1 - arg2;
                }

                return cdf;
        }

         // Three parameter log-normal probability
        public static double logNormalThreeParPDF(double alpha, double beta, double gamma,  double x){
            if(beta<0)throw new IllegalArgumentException("The parameter beta, " + beta + ", must be greater than or equal to zero");
            if(x<0){
                return 0.0D;
            }
            else{
                return Math.exp(-0.5D*Fmath.square(Math.log((x - alpha)/gamma)/beta))/((x - gamma)*beta*Math.sqrt(2.0D*Math.PI));
            }
        }


        // Returns an array of three parameter log-normal random deviates - clock seed
        public static double[] logNormalThreeParRand(double alpha, double beta, double gamma,  int n){
            if(n<=0)throw new IllegalArgumentException("The number of random deviates required, " + n + ", must be greater than zero");
            if(beta<0)throw new IllegalArgumentException("The parameter beta, " + beta + ", must be greater than or equal to zero");
            PsRandom psr = new PsRandom();
            return psr.logNormalThreeParArray(alpha, beta, gamma, n);
        }

         // Returns an array of three parameter log-normal random deviates - user supplied seed
        public static double[] logNormalThreeParRand(double alpha, double beta, double gamma,  int n, long seed){
            if(n<=0)throw new IllegalArgumentException("The number of random deviates required, " + n + ", must be greater than zero");
            if(beta<0)throw new IllegalArgumentException("The parameter beta, " + beta + ", must be greater than or equal to zero");
            PsRandom psr = new PsRandom(seed);
            return psr.logNormalThreeParArray(alpha, beta, gamma, n);
        }

        // Three parameter log-normal mean
        public static double logNormalThreeParMean(double alpha, double beta, double gamma){
                return gamma*Math.exp(beta*beta/2.0D) + alpha;
        }

        // Three parameter log-normal standard deviation
        public static double logNormalThreeParStandDev(double alpha, double beta, double gamma){
                double beta2 = beta*beta;
                return Math.sqrt((Math.exp(beta2) - 1.0D)*Math.exp(2.0D*Math.log(gamma) + beta2));
        }

        // Three parameter log-normal mode
        public static double logNormalThreeParMode(double alpha, double beta, double gamma){
            return gamma*Math.exp(- beta*beta) + alpha;
        }

        // Three parameter log-normal median
         public static double logNormalThreeParMedian(double alpha, double gamma){
            return gamma + alpha;
        }


        // LOGISTIC DISTRIBUTION

        // Logistic cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mu  =  location parameter, beta = scale parameter
        public static double logisticCDF(double mu, double beta, double upperlimit){
                return 0.5D*(1.0D + Math.tanh((upperlimit - mu)/(2.0D*beta)));
        }


        // Logistic cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        // mu  =  location parameter, beta = scale parameter
        public static double logisticProb(double mu, double beta, double upperlimit){
                return 0.5D*(1.0D + Math.tanh((upperlimit - mu)/(2.0D*beta)));
        }


        // Logistic cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mu  =  location parameter, beta = scale parameter
        public static double logisticCDF(double mu, double beta, double lowerlimit, double upperlimit){
                double arg1 = 0.5D*(1.0D + Math.tanh((lowerlimit - mu)/(2.0D*beta)));
                double arg2 = 0.5D*(1.0D + Math.tanh((upperlimit - mu)/(2.0D*beta)));
                return arg2 - arg1;
        }

        // Logistic cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        // mu  =  location parameter, beta = scale parameter
        public static double logisticProb(double mu, double beta, double lowerlimit, double upperlimit){
                double arg1 = 0.5D*(1.0D + Math.tanh((lowerlimit - mu)/(2.0D*beta)));
                double arg2 = 0.5D*(1.0D + Math.tanh((upperlimit - mu)/(2.0D*beta)));
                return arg2 - arg1;
        }

         // Logistic probability density function
        // mu  =  location parameter, beta = scale parameter
        public static double logisticPDF(double mu, double beta, double x){
                return Fmath.square(Fmath.sech((x - mu)/(2.0D*beta)))/(4.0D*beta);
        }
        // Logistic probability density function
        // mu  =  location parameter, beta = scale parameter
        public static double logistic(double mu, double beta, double x){
                return Fmath.square(Fmath.sech((x - mu)/(2.0D*beta)))/(4.0D*beta);
        }

        // Returns an array of logistic distribution random deviates - clock seed
        // mu  =  location parameter, beta = scale parameter
        public static double[] logisticRand(double mu, double beta, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                    ran[i] = 2.0D*beta*Fmath.atanh(2.0D*rr.nextDouble() - 1.0D) + mu;
                }
                return ran;
        }

        // Returns an array of Logistic random deviates - user provided seed
        // mu  =  location parameter, beta = scale parameter
        public static double[] logisticRand(double mu, double beta, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                    ran[i] = 2.0D*beta*Fmath.atanh(2.0D*rr.nextDouble() - 1.0D) + mu;
                }
                return ran;
        }

        // Logistic distribution mean
        public static double logisticMean(double mu){
                return mu;
        }


        // Logistic distribution standard deviation
        public static double logisticStandDev(double beta){
                return Math.sqrt(Fmath.square(Math.PI*beta)/3.0D);
        }

        // Logistic distribution mode
        public static double logisticMode(double mu){
                return mu;
        }

        // Logistic distribution median
        public static double logisticMedian(double mu){
                return mu;
        }


        // LORENTZIAN DISTRIBUTION

        // Lorentzian cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        public static double lorentzianProb(double mu, double gamma, double upperlimit){
                double arg = (upperlimit - mu)/(gamma/2.0D);
                return (1.0D/Math.PI)*(Math.atan(arg)+Math.PI/2.0);
        }
        // Lorentzian cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double lorentzianCDF(double mu, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (upperlimit - mu)/(gamma/2.0D);
                double arg2 = (lowerlimit - mu)/(gamma/2.0D);
                return (1.0D/Math.PI)*(Math.atan(arg1)-Math.atan(arg2));
        }

        // Lorentzian cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double lorentzianProb(double mu, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (upperlimit - mu)/(gamma/2.0D);
                double arg2 = (lowerlimit - mu)/(gamma/2.0D);
                return (1.0D/Math.PI)*(Math.atan(arg1)-Math.atan(arg2));
        }

        // Lorentzian probability
        public static double lorentzianPDF(double mu, double gamma, double x){
                double arg =gamma/2.0D;
                return (1.0D/Math.PI)*arg/(Fmath.square(mu-x)+arg*arg);
        }

        // Lorentzian probability
        public static double lorentzian(double mu, double gamma, double x){
                double arg =gamma/2.0D;
                return (1.0D/Math.PI)*arg/(Fmath.square(mu-x)+arg*arg);
        }


        // Returns an array of Lorentzian random deviates - clock seed
        // mu  =  the mean, gamma = half-height width, length of array
        public static double[] lorentzianRand(double mu, double gamma, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                    ran[i]=Math.tan((rr.nextDouble()-0.5)*Math.PI);
                    ran[i] = ran[i]*gamma/2.0D+mu;
                }
                return ran;
        }

        // Returns an array of Lorentzian random deviates - user provided seed
        // mu  =  the mean, gamma = half-height width, length of array
        public static double[] lorentzianRand(double mu, double gamma, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                    ran[i]=Math.tan((rr.nextDouble()-0.5)*Math.PI);
                    ran[i] = ran[i]*gamma/2.0D+mu;
                }
                return ran;
        }


        // POISSON DISTRIBUTION

        // Cumulative Poisson Probability Function
        // probability that a number of Poisson random events will occur between 0 and k (inclusive)
        // k is an integer greater than equal to 1
        // mean  = mean of the Poisson distribution
        public static double poissonCDF(int k, double mean){
                if(k<1)throw new IllegalArgumentException("k must be an integer greater than or equal to 1");
                return Stat.incompleteGammaComplementary((double) k, mean);
        }

        // Cumulative Poisson Probability Function
        // probability that a number of Poisson random events will occur between 0 and k (inclusive)
        // k is an integer greater than equal to 1
        // mean  = mean of the Poisson distribution
        public static double poissonProb(int k, double mean){
                if(k<1)throw new IllegalArgumentException("k must be an integer greater than or equal to 1");
                return Stat.incompleteGammaComplementary((double) k, mean);
        }

        // Poisson Probability Function
        // k is an integer greater than or equal to zero
        // mean  = mean of the Poisson distribution
        public static double poissonPDF(int k, double mean){
                if(k<0)throw new IllegalArgumentException("k must be an integer greater than or equal to 0");
                return Math.pow(mean, k)*Math.exp(-mean)/Stat.factorial((double)k);
        }

        // Poisson Probability Function
        // k is an integer greater than or equal to zero
        // mean  = mean of the Poisson distribution
        public static double poisson(int k, double mean){
                if(k<0)throw new IllegalArgumentException("k must be an integer greater than or equal to 0");
                return Math.pow(mean, k)*Math.exp(-mean)/Stat.factorial((double)k);
        }

        // Returns an array of Poisson random deviates - clock seed
        // mean  =  the mean,  n = length of array
        // follows the ideas of Numerical Recipes
        public static double[] poissonRand(double mean, int n){

                Random rr = new Random();
                double[] ran = poissonRandCalc(rr, mean, n);
                return ran;
        }

        // Returns an array of Poisson random deviates - user provided seed
        // mean  =  the mean,  n = length of array
        // follows the ideas of Numerical Recipes
        public static double[] poissonRand(double mean, int n, long seed){

                Random rr = new Random(seed);
                double[] ran = poissonRandCalc(rr, mean, n);
                return ran;
        }

        // Calculates and returns an array of Poisson random deviates
        private static double[] poissonRandCalc(Random rr, double mean, int n){
                double[] ran = new double[n];
                double oldm = -1.0D;
                double expt = 0.0D;
                double em = 0.0D;
                double term = 0.0D;
                double sq = 0.0D;
                double lnMean = 0.0D;
                double yDev = 0.0D;

                if(mean < 12.0D){
                    for(int i=0; i<n; i++){
                        if(mean != oldm){
                            oldm = mean;
                            expt = Math.exp(-mean);
                        }
                        em = -1.0D;
                        term = 1.0D;
                        do{
                            ++em;
                            term *= rr.nextDouble();
                        }while(term>expt);
                        ran[i] = em;
                    }
                }
                else{
                    for(int i=0; i<n; i++){
                        if(mean != oldm){
                            oldm = mean;
                            sq = Math.sqrt(2.0D*mean);
                            lnMean = Math.log(mean);
                            expt = lnMean - Stat.logGamma(mean+1.0D);
                        }
                        do{
                            do{
                                yDev = Math.tan(Math.PI*rr.nextDouble());
                                em = sq*yDev+mean;
                            }while(em<0.0D);
                            em = Math.floor(em);
                            term = 0.9D*(1.0D+yDev*yDev)*Math.exp(em*lnMean - Stat.logGamma(em+1.0D)-expt);
                        }while(rr.nextDouble()>term);
                        ran[i] = em;
                    }
                }
                return ran;
        }


        // CHI SQUARE DISTRIBUTION AND CHI SQUARE FUNCTIONS

        // Chi-Square Cumulative Distribution Function
        // probability that an observed chi-square value for a correct model should be less than chiSquare
        // nu  =  the degrees of freedom
        public static double chiSquareCDF(double chiSquare, int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                return Stat.incompleteGamma((double)nu/2.0D, chiSquare/2.0D);
        }

        // retained for compatability
        public static double chiSquareProb(double chiSquare, int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                return Stat.incompleteGamma((double)nu/2.0D, chiSquare/2.0D);
        }

        // Chi-Square Probability Density Function
        // nu  =  the degrees of freedom
        public static double chiSquarePDF(double chiSquare, int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                double dnu = (double) nu;
                return Math.pow(0.5D, dnu/2.0D)*Math.pow(chiSquare, dnu/2.0D - 1.0D)*Math.exp(-chiSquare/2.0D)/Stat.gammaFunction(dnu/2.0D);
        }

        // Returns an array of Chi-Square random deviates - clock seed
        public static double[] chiSquareRand(int nu, int n){
            if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
            PsRandom psr = new PsRandom();
            return psr.chiSquareArray(nu, n);
        }


        // Returns an array of Chi-Square random deviates - user supplied seed
        public static double[] chiSquareRand(int nu, int n, long seed){
            if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
            PsRandom psr = new PsRandom(seed);
            return psr.chiSquareArray(nu, n);
        }

        // Chi-Square Distribution Mean
        // nu  =  the degrees of freedom
        public static double chiSquareMean(int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                return (double)nu;
        }

        // Chi-Square Distribution Mean
        // nu  =  the degrees of freedom
        public static double chiSquareMode(int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                double mode = 0.0D;
                if(nu>=2)mode = (double)nu - 2.0D;
                return mode;
        }

        // Chi-Square Distribution Standard Deviation
        // nu  =  the degrees of freedom
        public static double chiSquareStandDev(int nu){
                if(nu<=0)throw new IllegalArgumentException("The degrees of freedom [nu], " + nu + ", must be greater than zero");
                double dnu = (double) nu;
                return Math.sqrt(2.0D*dnu);
        }

        // Chi-Square Statistic
        public static double chiSquare(double[] observed, double[] expected, double[] variance){
            int nObs = observed.length;
            int nExp = expected.length;
            int nVar = variance.length;
            if(nObs!=nExp)throw new IllegalArgumentException("observed array length does not equal the expected array length");
            if(nObs!=nVar)throw new IllegalArgumentException("observed array length does not equal the variance array length");
            double chi = 0.0D;
            for(int i=0; i<nObs; i++){
                chi += Fmath.square(observed[i]-expected[i])/variance[i];
            }
            return chi;
        }

        // Chi-Square Statistic for Poisson distribution for frequency data
        // and Poisson distribution for each bin
        // double arguments
        public static double chiSquareFreq(double[] observedFreq, double[] expectedFreq){
            int nObs = observedFreq.length;
            int nExp = expectedFreq.length;
            if(nObs!=nExp)throw new IllegalArgumentException("observed array length does not equal the expected array length");
            double chi = 0.0D;
            for(int i=0; i<nObs; i++){
                chi += Fmath.square(observedFreq[i]-expectedFreq[i])/expectedFreq[i];
            }
            return chi;
        }

        // Chi-Square Statistic for Poisson distribution for frequency data
        // and Poisson distribution for each bin
        // int arguments
        public static double chiSquareFreq(int[] observedFreq, int[] expectedFreq){
            int nObs = observedFreq.length;
            int nExp = expectedFreq.length;
            if(nObs!=nExp)throw new IllegalArgumentException("observed array length does not equal the expected array length");
            double[] observ = new double[nObs];
            double[] expect = new double[nObs];
            for(int i=0; i<nObs; i++){
                observ[i] = observedFreq[i];
                expect[i] = expectedFreq[i];
            }

            return chiSquareFreq(observ, expect);
        }


        // BINOMIAL DISTRIBUTION AND BINOMIAL COEFFICIENTS

        // Returns the binomial cumulative probabilty
        public static double binomialCDF(double p, int n, int k){
                if(p<0.0D || p>1.0D)throw new IllegalArgumentException("\np must lie between 0 and 1");
                if(k<0 || n<0)throw new IllegalArgumentException("\nn and k must be greater than or equal to zero");
                if(k>n)throw new IllegalArgumentException("\nk is greater than n");
                return Stat.regularisedBetaFunction(k, n-k+1, p);
        }
        // Returns the binomial cumulative probabilty
        public static double binomialProb(double p, int n, int k){
                if(p<0.0D || p>1.0D)throw new IllegalArgumentException("\np must lie between 0 and 1");
                if(k<0 || n<0)throw new IllegalArgumentException("\nn and k must be greater than or equal to zero");
                if(k>n)throw new IllegalArgumentException("\nk is greater than n");
                return Stat.regularisedBetaFunction(k, n-k+1, p);
        }

         // Returns a binomial mass probabilty function
        public static double binomialPDF(double p, int n, int k){
                if(k<0 || n<0)throw new IllegalArgumentException("\nn and k must be greater than or equal to zero");
                if(k>n)throw new IllegalArgumentException("\nk is greater than n");
                return Math.floor(0.5D + Math.exp(Stat.logFactorial(n) - Stat.logFactorial(k) - Stat.logFactorial(n-k)))*Math.pow(p, k)*Math.pow(1.0D - p, n - k);
        }

        // Returns a binomial mass probabilty function
        public static double binomial(double p, int n, int k){
                if(k<0 || n<0)throw new IllegalArgumentException("\nn and k must be greater than or equal to zero");
                if(k>n)throw new IllegalArgumentException("\nk is greater than n");
                return Math.floor(0.5D + Math.exp(Stat.logFactorial(n) - Stat.logFactorial(k) - Stat.logFactorial(n-k)))*Math.pow(p, k)*Math.pow(1.0D - p, n - k);
        }

        // Returns a binomial Coefficient as a double
        public static double binomialCoeff(int n, int k){
                if(k<0 || n<0)throw new IllegalArgumentException("\nn and k must be greater than or equal to zero");
                if(k>n)throw new IllegalArgumentException("\nk is greater than n");
                return Math.floor(0.5D + Math.exp(Stat.logFactorial(n) - Stat.logFactorial(k) - Stat.logFactorial(n-k)));
        }

        // Returns an array of n Binomial pseudorandom deviates from a binomial - clock seed
        //  distribution of nTrial trials each of probablity, prob,
        //  after 	bndlev 	Numerical Recipes in C - W.H. Press et al. (Cambridge)
        //		            2nd edition 1992 p295.
        public double[] binomialRand(double prob, int nTrials, int n){

            if(nTrials<n)throw new IllegalArgumentException("Number of deviates requested, " + n + ", must be less than the number of trials, " + nTrials);
            if(prob<0.0D || prob>1.0D)throw new IllegalArgumentException("The probablity provided, " + prob + ", must lie between 0 and 1)");

            double[] ran = new double[n];                   // array of deviates to be returned
            Random rr = new Random();                       // instance of Random

	        double binomialDeviate = 0.0D;                  // the binomial deviate to be returned
	        double deviateMean = 0.0D;                      // mean of deviate to be produced
	        double testDeviate = 0.0D;                      // test deviate
	        double workingProb = 0.0;                       // working value of the probability
	        double logProb = 0.0;                           // working value of the probability
	        double probOld = -1.0D;                         // previous value of the working probability
	        double probC = -1.0D;                           // complementary value of the working probability
	        double logProbC = -1.0D;                        // log of the complementary value of the working probability
	        int nOld= -1;                                   // previous value of trials counter
	        double enTrials = 0.0D;                         // (double) trials counter
	        double oldGamma = 0.0D;                         // a previous log Gamma function value
	        double tanW = 0.0D;                             // a working tangent
	        double hold0 = 0.0D;                            // a working holding variable
	        int jj;                                         // counter

            double probOriginalValue = prob;
            for(int i=0; i<n; i++){
                prob = probOriginalValue;
	            workingProb=(prob <= 0.5D ? prob : 1.0-prob);    // distribution invariant on swapping prob for 1 - prob
	            deviateMean = nTrials*workingProb;

	            if(nTrials < 25) {
	                // if number of trials greater than 25 use direct method
		            binomialDeviate=0.0D;
		            for(jj=1;jj<=nTrials;jj++)if (rr.nextDouble() < workingProb) ++binomialDeviate;
	            }
	            else if(deviateMean < 1.0D) {
	                // if fewer than 1 out of 25 events - Poisson approximation is accurate
		            double expOfMean=Math.exp(-deviateMean);
		            testDeviate=1.0D;
		            for (jj=0;jj<=nTrials;jj++) {
			            testDeviate *= rr.nextDouble();
			            if (testDeviate < expOfMean) break;
		            }
		            binomialDeviate=(jj <= nTrials ? jj : nTrials);

	            }
	            else{
	                // use rejection method
		            if(nTrials != nOld) {
		                // if nTrials has changed compute useful quantities
			            enTrials = (double)nTrials;
			            oldGamma = Stat.logGamma(enTrials + 1.0D);
			            nOld = nTrials;
		            }
		            if(workingProb != probOld) {
		                // if workingProb has changed compute useful quantities
                        probC = 1.0 - workingProb;
			            logProb = Math.log(workingProb);
			            logProbC = Math.log(probC);
			            probOld = workingProb;
		            }

		            double sq = Math.sqrt(2.0*deviateMean*probC);
		            do{
			            do{
				            double angle = Math.PI*rr.nextDouble();
				            tanW = Math.tan(angle);
				            hold0 = sq*tanW + deviateMean;
			            }while(hold0 < 0.0D || hold0 >= (enTrials + 1.0D));   //rejection test
			            hold0 = Math.floor(hold0);                              // integer value distribution
			            testDeviate = 1.2D*sq*(1.0D + tanW*tanW)*Math.exp(oldGamma - Stat.logGamma(hold0 + 1.0D) - Stat.logGamma(enTrials - hold0 + 1.0D) + hold0*logProb + (enTrials - hold0)*logProbC);
		            }while(rr.nextDouble() > testDeviate);                         // rejection test
		            binomialDeviate=hold0;
	            }

	            if(workingProb != prob) binomialDeviate = nTrials - binomialDeviate;       // symmetry transformation

	            ran[i] = binomialDeviate;
	        }

	        return ran;
        }

        // Returns an array of n Binomial pseudorandom deviates from a binomial - user supplied seed
        //  distribution of nTrial trials each of probablity, prob,
        //  after 	bndlev 	Numerical Recipes in C - W.H. Press et al. (Cambridge)
        //		            2nd edition 1992 p295.
        public double[] binomialRand(double prob, int nTrials, int n, long seed){

            if(nTrials<n)throw new IllegalArgumentException("Number of deviates requested, " + n + ", must be less than the number of trials, " + nTrials);
            if(prob<0.0D || prob>1.0D)throw new IllegalArgumentException("The probablity provided, " + prob + ", must lie between 0 and 1)");

            double[] ran = new double[n];                   // array of deviates to be returned
            Random rr = new Random(seed);                       // instance of Random

	        double binomialDeviate = 0.0D;                  // the binomial deviate to be returned
	        double deviateMean = 0.0D;                      // mean of deviate to be produced
	        double testDeviate = 0.0D;                      // test deviate
	        double workingProb = 0.0;                       // working value of the probability
	        double logProb = 0.0;                           // working value of the probability
	        double probOld = -1.0D;                         // previous value of the working probability
	        double probC = -1.0D;                           // complementary value of the working probability
	        double logProbC = -1.0D;                        // log of the complementary value of the working probability
	        int nOld= -1;                                   // previous value of trials counter
	        double enTrials = 0.0D;                         // (double) trials counter
	        double oldGamma = 0.0D;                         // a previous log Gamma function value
	        double tanW = 0.0D;                             // a working tangent
	        double hold0 = 0.0D;                            // a working holding variable
	        int jj;                                         // counter

            double probOriginalValue = prob;
            for(int i=0; i<n; i++){
                prob = probOriginalValue;
	            workingProb=(prob <= 0.5D ? prob : 1.0-prob);    // distribution invariant on swapping prob for 1 - prob
	            deviateMean = nTrials*workingProb;

	            if(nTrials < 25) {
	                // if number of trials greater than 25 use direct method
		            binomialDeviate=0.0D;
		            for(jj=1;jj<=nTrials;jj++)if (rr.nextDouble() < workingProb) ++binomialDeviate;
	            }
	            else if(deviateMean < 1.0D) {
	                // if fewer than 1 out of 25 events - Poisson approximation is accurate
		            double expOfMean=Math.exp(-deviateMean);
		            testDeviate=1.0D;
		            for (jj=0;jj<=nTrials;jj++) {
			            testDeviate *= rr.nextDouble();
			            if (testDeviate < expOfMean) break;
		            }
		            binomialDeviate=(jj <= nTrials ? jj : nTrials);

	            }
	            else{
	                // use rejection method
		            if(nTrials != nOld) {
		                // if nTrials has changed compute useful quantities
			            enTrials = (double)nTrials;
			            oldGamma = Stat.logGamma(enTrials + 1.0D);
			            nOld = nTrials;
		            }
		            if(workingProb != probOld) {
		                // if workingProb has changed compute useful quantities
                        probC = 1.0 - workingProb;
			            logProb = Math.log(workingProb);
			            logProbC = Math.log(probC);
			            probOld = workingProb;
		            }

		            double sq = Math.sqrt(2.0*deviateMean*probC);
		            do{
			            do{
				            double angle = Math.PI*rr.nextDouble();
				            tanW = Math.tan(angle);
				            hold0 = sq*tanW + deviateMean;
			            }while(hold0 < 0.0D || hold0 >= (enTrials + 1.0D));   //rejection test
			            hold0 = Math.floor(hold0);                              // integer value distribution
			            testDeviate = 1.2D*sq*(1.0D + tanW*tanW)*Math.exp(oldGamma - Stat.logGamma(hold0 + 1.0D) - Stat.logGamma(enTrials - hold0 + 1.0D) + hold0*logProb + (enTrials - hold0)*logProbC);
		            }while(rr.nextDouble() > testDeviate);                         // rejection test
		            binomialDeviate=hold0;
	            }

	            if(workingProb != prob) binomialDeviate = nTrials - binomialDeviate;       // symmetry transformation

	            ran[i] = binomialDeviate;
	        }

	        return ran;
        }


        // F-DISTRIBUTION AND F-TEST

        // Returns the F-distribution probabilty for degrees of freedom df1, df2
        // F ratio provided
        public static double fCompCDF(double fValue, int df1, int df2){
            if(df1<=0)throw new IllegalArgumentException("the degrees of freedom, nu1, " + df1 + ", must be greater than zero");
            if(df2<=0)throw new IllegalArgumentException("the degrees of freedom, nu2, " + df2 + ", must be greater than zero");
            if(fValue<0)throw new IllegalArgumentException("the F-ratio, " + fValue + ", must be greater than or equal to zero");
            double ddf1 = (double)df1;
            double ddf2 = (double)df2;
            double x = ddf2/(ddf2+ddf1*fValue);
            return Stat.regularisedBetaFunction(df2/2.0D, df1/2.0D, x);
        }

        // retained fot compatibility
        public static double fTestProb(double fValue, int df1, int df2){
            if(df1<=0)throw new IllegalArgumentException("the degrees of freedom, nu1, " + df1 + ", must be greater than zero");
            if(df2<=0)throw new IllegalArgumentException("the degrees of freedom, nu2, " + df2 + ", must be greater than zero");
            if(fValue<0)throw new IllegalArgumentException("the F-ratio, " + fValue + ", must be greater than or equal to zero");
            double ddf1 = (double)df1;
            double ddf2 = (double)df2;
            double x = ddf2/(ddf2+ddf1*fValue);
            return Stat.regularisedBetaFunction(df2/2.0D, df1/2.0D, x);
        }

        // Returns the F-distribution probabilty for degrees of freedom df1, df2
        // numerator and denominator variances provided
        public static double fCompCDF(double var1, int df1, double var2, int df2){
            if(df1<=0)throw new IllegalArgumentException("the degrees of freedom, nu1, " + df1 + ", must be greater than zero");
            if(df2<=0)throw new IllegalArgumentException("the degrees of freedom, nu2, " + df2 + ", must be greater than zero");
            if(var1<0)throw new IllegalArgumentException("the variance, var1" + var1 + ", must be greater than or equal to zero");
            if(var1<=0)throw new IllegalArgumentException("the variance, var2" + var2 + ", must be greater than zero");
            double fValue = var1/var2;
            double ddf1 = (double)df1;
            double ddf2 = (double)df2;
            double x = ddf2/(ddf2+ddf1*fValue);
            return Stat.regularisedBetaFunction(df2/2.0D, df1/2.0D, x);
        }

        // retained fot compatibility
        public static double fTestProb(double var1, int df1, double var2, int df2){
            if(df1<=0)throw new IllegalArgumentException("the degrees of freedom, nu1, " + df1 + ", must be greater than zero");
            if(df2<=0)throw new IllegalArgumentException("the degrees of freedom, nu2, " + df2 + ", must be greater than zero");
            if(var1<0)throw new IllegalArgumentException("the variance, var1" + var1 + ", must be greater than or equal to zero");
            if(var1<=0)throw new IllegalArgumentException("the variance, var2" + var2 + ", must be greater than zero");
            double fValue = var1/var2;
            double ddf1 = (double)df1;
            double ddf2 = (double)df2;
            double x = ddf2/(ddf2+ddf1*fValue);
            return Stat.regularisedBetaFunction(df2/2.0D, df1/2.0D, x);
        }

        // Returns the F-test value corresponding to a F-distribution probabilty, fProb,
        //   for degrees of freedom df1, df2
        public static double fTestValueGivenFprob(double fProb, int df1, int df2){

            // Create an array F-test value array
            int fTestsNum = 100;    // length of array
            double[] fTestValues = new double[fTestsNum];
            fTestValues[0]=0.0001D;             // lowest array value
            fTestValues[fTestsNum-1]=10000.0D;  // highest array value
            // calculate array increment - log scale
            double diff = (Fmath.log10(fTestValues[fTestsNum-1])-Fmath.log10(fTestValues[0]))/(fTestsNum-1);
            // Fill array
            for(int i=1; i<fTestsNum-1; i++){
                fTestValues[i] = Math.pow(10.0D,(Fmath.log10(fTestValues[i-1])+diff));
            }

            // calculate F test probability array corresponding to F-test value array
            double[] fTestProb = new double[fTestsNum];
            for(int i=0; i<fTestsNum; i++){
                fTestProb[i] = Stat.fTestProb(fTestValues[i], df1, df2);
            }

            // calculate F-test value for provided probability
            // using bisection procedure
            double fTest0 = 0.0D;
            double fTest1 = 0.0D;
            double fTest2 = 0.0D;

            // find bracket for the F-test probabilities and calculate F-Test value from above arrays
            boolean test0 = true;
            boolean test1 = true;
            int i=0;
            int endTest=0;
            while(test0){
                if(fProb==fTestProb[i]){
                    fTest0=fTestValues[i];
                    test0=false;
                    test1=false;
                }
                else{
                    if(fProb>fTestProb[i]){
                        test0=false;
                        if(i>0){
                            fTest1=fTestValues[i-1];
                            fTest2=fTestValues[i];
                            endTest=-1;
                        }
                        else{
                            fTest1=fTestValues[i]/10.0D;
                            fTest2=fTestValues[i];
                        }
                    }
                    else{
                        i++;
                        if(i>fTestsNum-1){
                            test0=false;
                            fTest1=fTestValues[i-1];
                            fTest2=10.0D*fTestValues[i-1];
                            endTest=1;
                        }
                    }
                }
            }

            // call bisection method
            if(test1)fTest0=fTestBisect(fProb, fTest1, fTest2, df1, df2, endTest);

            return fTest0;
        }

        // Bisection procedure for calculating and F-test value corresponding
        //   to a given F-test probability
        private static double fTestBisect(double fProb, double fTestLow, double fTestHigh, int df1, int df2, int endTest){

            double funcLow = fProb - Stat.fTestProb(fTestLow, df1, df2);
            double funcHigh = fProb - Stat.fTestProb(fTestHigh, df1, df2);
            double fTestMid = 0.0D;
            double funcMid = 0.0;
            int nExtensions = 0;
            int nIter = 1000;           // iterations allowed
            double check = fProb*1e-6;  // tolerance for bisection
            boolean test0 = true;       // test for extending bracket
            boolean test1 = true;       // test for bisection procedure
            while(test0){
                if(funcLow*funcHigh>0.0D){
                    if(endTest<0){
                        nExtensions++;
                        if(nExtensions>100){
                            System.out.println("Class: Stats\nMethod: fTestBisect\nProbability higher than range covered\nF-test value is less than "+fTestLow);
                            System.out.println("This value was returned");
                            fTestMid=fTestLow;
                            test0=false;
                            test1=false;
                        }
                        fTestLow /= 10.0D;
                        funcLow = fProb - Stat.fTestProb(fTestLow, df1, df2);
                    }
                    else{
                        nExtensions++;
                        if(nExtensions>100){
                            System.out.println("Class: Stats\nMethod: fTestBisect\nProbability lower than range covered\nF-test value is greater than "+fTestHigh);
                            System.out.println("This value was returned");
                            fTestMid=fTestHigh;
                            test0=false;
                            test1=false;
                        }
                        fTestHigh *= 10.0D;
                        funcHigh = fProb - Stat.fTestProb(fTestHigh, df1, df2);
                    }
                }
                else{
                    test0=false;
                }

                int i=0;
                while(test1){
                    fTestMid = (fTestLow+fTestHigh)/2.0D;
                    funcMid = fProb - Stat.fTestProb(fTestMid, df1, df2);
                    if(Math.abs(funcMid)<check){
                        test1=false;
                    }
                    else{
                        i++;
                        if(i>nIter){
                            System.out.println("Class: Stats\nMethod: fTestBisect\nmaximum number of iterations exceeded\ncurrent value of F-test value returned");
                            test1=false;
                        }
                        if(funcMid*funcHigh>0){
                            funcHigh=funcMid;
                            fTestHigh=fTestMid;
                        }
                        else{
                            funcLow=funcMid;
                            fTestLow=fTestMid;
                        }
                    }
                }
            }
            return fTestMid;
        }

        // F-distribution pdf
        public double fPDF(double fValue, int nu1, int nu2){
            double numer = Math.pow(nu1*fValue, nu1)*Math.pow(nu2, nu2);
            double dnu1 = (double)nu1;
            double dnu2 = (double)nu2;
            numer /= Math.pow(dnu1*fValue+dnu2, dnu1+dnu2);
            numer = Math.sqrt(numer);
            double denom = fValue*Stat.betaFunction(dnu1/2.0D, dnu2/2.0D);
            return numer/denom;
        }

        public double fPDF(double var1, int nu1, double var2, int nu2){
            return fPDF(var1/var2, nu1, nu2);
        }

        // Returns an array of F-distribution random deviates - clock seed
        public static double[] fRand(int nu1, int nu2, int n){
            if(nu1<=0)throw new IllegalArgumentException("The degrees of freedom [nu1], " + nu1 + ", must be greater than zero");
            if(nu2<=0)throw new IllegalArgumentException("The degrees of freedom [nu2], " + nu2 + ", must be greater than zero");
            PsRandom psr = new PsRandom();
            return psr.fArray(nu1, nu2, n);
        }

        // Returns an array of F-distribution random deviates - user supplied seed
        public static double[] fRand(int nu1, int nu2, int n, long seed){
            if(nu1<=0)throw new IllegalArgumentException("The degrees of freedom [nu1], " + nu1 + ", must be greater than zero");
            if(nu2<=0)throw new IllegalArgumentException("The degrees of freedom [nu2], " + nu2 + ", must be greater than zero");
             PsRandom psr = new PsRandom(seed);
            return psr.fArray(nu1, nu2, n);
        }

        // STUDENT'S T DISTRIBUTION

        // Returns the Student's t probability density function
        public static double studentT(double tValue, int df){
            double ddf = (double)df;
            double dfterm = (ddf + 1.0D)/2.0D;
            return ((Stat.gamma(dfterm)/Stat.gamma(ddf/2))/Math.sqrt(ddf*Math.PI))*Math.pow(1.0D + tValue*tValue/ddf, -dfterm);
        }

        // Returns the Student's t probability density function
        public static double studentTpdf(double tValue, int df){
            double ddf = (double)df;
            double dfterm = (ddf + 1.0D)/2.0D;
            return ((Stat.gamma(dfterm)/Stat.gamma(ddf/2))/Math.sqrt(ddf*Math.PI))*Math.pow(1.0D + tValue*tValue/ddf, -dfterm);
        }

        // Returns the Student's t probability density function
        public static double studentTPDF(double tValue, int df){
            double ddf = (double)df;
            double dfterm = (ddf + 1.0D)/2.0D;
            return ((Stat.gamma(dfterm)/Stat.gamma(ddf/2))/Math.sqrt(ddf*Math.PI))*Math.pow(1.0D + tValue*tValue/ddf, -dfterm);
        }

         // Returns the Student's t cumulative distribution function probability
        public static double studentTProb(double tValue, int df){
            double ddf = (double)df;
            double x = ddf/(ddf+tValue*tValue);
            return 0.5D*(1.0D + (Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, 1) - Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, x))*Fmath.sign(tValue));
        }

        // Returns the Student's t cumulative distribution function probability
        public static double studentTcdf(double tValue, int df){
            double ddf = (double)df;
            double x = ddf/(ddf+tValue*tValue);
            return 0.5D*(1.0D + (Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, 1) - Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, x))*Fmath.sign(tValue));
        }

        // Returns the Student's t cumulative distribution function probability
        public static double studentTCDF(double tValue, int df){
            double ddf = (double)df;
            double x = ddf/(ddf+tValue*tValue);
            return 0.5D*(1.0D + (Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, 1) - Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, x))*Fmath.sign(tValue));
        }

        // Returns the Student's t mean,  df = degrees of freedom
        public static double studentTmean(int df){
            double mean = Double.NaN;    // mean undefined for df = 1
            if(df>1)mean = 0.0D;
            return mean;
        }

        // Returns the Student's t median
        public static double studentTmedian(){
            return 0.0;
        }

        // Returns the Student's t mode
        public static double studentTmode(){
            return 0.0;
        }

        // Returns the Student's t standard deviation,  df = degrees of freedom
        public static double studentTstandDev(int df){
            double standDev = Double.POSITIVE_INFINITY;
            if(df>2)standDev = Math.sqrt(df/(1 - df));
            return standDev;
        }

        // Returns the A(t|n) distribution probabilty
        public static double probAtn(double tValue, int df){
            double ddf = (double)df;
            double x = ddf/(ddf+tValue*tValue);
            return 1.0D - Stat.regularisedBetaFunction(ddf/2.0D, 0.5D, x);
        }

        // Returns an array of Student's t random deviates - clock seed
        // nu  =  the degrees of freedom
        public static double[] studentTRand(int nu, int n){
            PsRandom psr = new PsRandom();
            return psr.studentTarray(nu, n);
        }

        // Returns an array of Student's t random deviates - clock seed
        // nu  =  the degrees of freedom
        public static double[] studentTrand(int nu, int n){
            PsRandom psr = new PsRandom();
            return psr.studentTarray(nu, n);
        }

        // Returns an array of a Student's t random deviates - user supplied seed
        // nu  =  the degrees of freedom
        public static double[] studentTrand(int nu, int n, long seed){
            PsRandom psr = new PsRandom(seed);
            return psr.studentTarray(nu, n);
        }

        // Returns an array of a Student's t random deviates - user supplied seed
        // nu  =  the degrees of freedom
        public static double[] studentTRand(int nu, int n, long seed){
            PsRandom psr = new PsRandom(seed);
            return psr.studentTarray(nu, n);
        }


        // GUMBEL (TYPE I EXTREME VALUE) DISTRIBUTION

        // Minimum Gumbel cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        public static double gumbelMinProbCDF(double mu, double sigma, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg = -(upperlimit - mu)/sigma;
                return Math.exp(-Math.exp(arg));
        }

        // Minimum Gumbel cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        public static double gumbelMinProb(double mu, double sigma, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg = -(upperlimit - mu)/sigma;
                return Math.exp(-Math.exp(arg));
        }

        // Maximum Gumbel cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        public static double gumbelMaxCDF(double mu, double sigma, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg = -(upperlimit - mu)/sigma;
                return 1.0D-Math.exp(-Math.exp(arg));
        }

        // Maximum Gumbel cumulative distribution function
        // probability that a variate will assume a value less than the upperlimit
        public static double gumbelMaxProb(double mu, double sigma, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg = -(upperlimit - mu)/sigma;
                return 1.0D-Math.exp(-Math.exp(arg));
        }


         // Minimum Gumbel cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double gumbelMinCDF(double mu, double sigma, double lowerlimit, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg1 = -(lowerlimit - mu)/sigma;
                double arg2 = -(upperlimit - mu)/sigma;
                double term1 = Math.exp(-Math.exp(arg1));
                double term2 = Math.exp(-Math.exp(arg2));
                return term2-term1;
        }

        // Minimum Gumbel cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double gumbelMinProb(double mu, double sigma, double lowerlimit, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg1 = -(lowerlimit - mu)/sigma;
                double arg2 = -(upperlimit - mu)/sigma;
                double term1 = Math.exp(-Math.exp(arg1));
                double term2 = Math.exp(-Math.exp(arg2));
                return term2-term1;
        }

         // Maximum Gumbel cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double gumbelMaxCDF(double mu, double sigma, double lowerlimit, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = -Math.exp(-Math.exp(arg1));
                double term2 = -Math.exp(-Math.exp(arg2));
                return term2-term1;
        }

        // Maximum Gumbel cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double gumbelMaxProb(double mu, double sigma, double lowerlimit, double upperlimit){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = -Math.exp(-Math.exp(arg1));
                double term2 = -Math.exp(-Math.exp(arg2));
                return term2-term1;
        }

        // Minimum Gumbel probability
        public static double gumbelMinPDF(double mu,double sigma, double x){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg =(x-mu)/sigma;
                return (1.0D/sigma)*Math.exp(arg)*Math.exp(-Math.exp(arg));
        }

                // Minimum Gumbel probability
        public static double gumbelMin(double mu,double sigma, double x){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg =(x-mu)/sigma;
                return (1.0D/sigma)*Math.exp(arg)*Math.exp(-Math.exp(arg));
        }

         // Maximum Gumbel probability
        public static double gumbelMaxPDF(double mu,double sigma, double x){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg =-(x-mu)/sigma;
                return (1.0D/sigma)*Math.exp(arg)*Math.exp(-Math.exp(arg));
        }

        // Maximum Gumbel probability
        public static double gumbelMax(double mu,double sigma, double x){
                if(sigma<0.0D)throw new IllegalArgumentException("sigma must be positive");
                double arg =-(x-mu)/sigma;
                return (1.0D/sigma)*Math.exp(arg)*Math.exp(-Math.exp(arg));
        }

        // Returns an array of minimal Gumbel (Type I EVD) random deviates - clock seed
        // mu  =  location parameter, sigma = scale parameter, n = length of array
        public static double[] gumbelMinRand(double mu, double sigma,  int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = Math.log(Math.log(1.0D/(1.0D-rr.nextDouble())))*sigma+mu;
                }
                return ran;
        }

        // Returns an array of minimal Gumbel (Type I EVD) random deviates - user supplied seed
        // mu  =  location parameter, sigma = scale parameter, n = length of array
        public static double[] gumbelMinRand(double mu, double sigma,  int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = Math.log(Math.log(1.0D/(1.0D-rr.nextDouble())))*sigma+mu;
                }
                return ran;
        }

        // Returns an array of maximal Gumbel (Type I EVD) random deviates - clock seed
        // mu  =  location parameter, sigma = scale parameter, n = length of array
        public static double[] gumbelMaxRand(double mu, double sigma,  int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = mu-Math.log(Math.log(1.0D/(1.0D-rr.nextDouble())))*sigma;
                }
                return ran;
        }

       // Returns an array of maximal Gumbel (Type I EVD) random deviates - user supplied seed
        // mu  =  location parameter, sigma = scale parameter, n = length of array
        public static double[] gumbelMaxRand(double mu, double sigma,  int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = mu-Math.log(Math.log(1.0D/(1.0D-rr.nextDouble())))*sigma;
                }
                return ran;
        }

        // Minimum Gumbel mean
        public static double gumbelMinMean(double mu,double sigma){
                return mu - sigma*Fmath.EULER_CONSTANT_GAMMA;
        }

        // Maximum Gumbel mean
        public static double gumbelMaxMean(double mu,double sigma){
                return mu + sigma*Fmath.EULER_CONSTANT_GAMMA;
        }

        // Minimum Gumbel standard deviation
        public static double gumbelMinStandDev(double sigma){
                return sigma*Math.PI/Math.sqrt(6.0D);
        }

        // Maximum Gumbel standard deviation
        public static double gumbelMaxStandDev(double sigma){
                return sigma*Math.PI/Math.sqrt(6.0D);
        }

        // Minimum Gumbel mode
        public static double gumbelMinMode(double mu,double sigma){
            return mu;
        }

        // Maximum Gumbel mode
        public static double gumbelMaxMode(double mu,double sigma){
            return mu;
        }

        // Minimum Gumbel median
        public static double gumbelMinMedian(double mu,double sigma){
            return mu + sigma*Math.log(Math.log(2.0D));
        }

        // Maximum Gumbel median
        public static double gumbelMaxMedian(double mu,double sigma){
            return mu - sigma*Math.log(Math.log(2.0D));
        }


        // FRECHET (TYPE II EXTREME VALUE) DISTRIBUTION

        // Frechet cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double frechetProb(double mu, double sigma, double gamma, double upperlimit){
                double arg = (upperlimit - mu)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = Math.exp(-Math.pow(arg, -gamma));
                return y;
        }


        // Frechet cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double frechetCDF(double mu, double sigma, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = Math.exp(-Math.pow(arg1, -gamma));
                if(arg2>=0.0D)term2 = Math.exp(-Math.pow(arg2, -gamma));
                return term2-term1;
        }

        // Frechet cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double frechetProb(double mu, double sigma, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = Math.exp(-Math.pow(arg1, -gamma));
                if(arg2>=0.0D)term2 = Math.exp(-Math.pow(arg2, -gamma));
                return term2-term1;
        }

        // Frechet probability
        public static double frechetPDF(double mu,double sigma, double gamma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (gamma/sigma)*Math.pow(arg, -gamma-1.0D)*Math.exp(-Math.pow(arg, -gamma));
                }
                return y;
        }

        // Frechet probability
        public static double frechet(double mu,double sigma, double gamma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (gamma/sigma)*Math.pow(arg, -gamma-1.0D)*Math.exp(-Math.pow(arg, -gamma));
                }
                return y;
        }

        // Frechet mean
        public static double frechetMean(double mu,double sigma, double gamma){
                double y = Double.NaN;
                if(gamma>1.0D){
                    y = mu + sigma*Stat.gamma(1.0D-1.0D/gamma);
                }
                return y;
        }

        // Frechet standard deviation
        public static double frechetStandDev(double sigma, double gamma){
                double y = Double.NaN;
                if(gamma>2.0D){
                    y = Stat.gamma(1.0D-2.0D/gamma)-Fmath.square(Stat.gamma(1.0D-1.0D/gamma));
                    y = sigma*Math.sqrt(y);
                }
                return y;
        }

        // Frechet mode
        public static double frechetMode(double mu,double sigma, double gamma){
                return mu + sigma*Math.pow(gamma/(1.0D+gamma), 1.0D/gamma);
         }

        // Returns an array of Frechet (Type II EVD) random deviates - clock seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] frechetRand(double mu, double sigma, double gamma, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow((1.0D/(Math.log(1.0D/rr.nextDouble()))),1.0D/gamma)*sigma + mu;
                }
                return ran;
        }

        // Returns an array of Frechet (Type II EVD) random deviates - user supplied seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] frechetRand(double mu, double sigma, double gamma, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow((1.0D/(Math.log(1.0D/rr.nextDouble()))),1.0D/gamma)*sigma + mu;
                }
                return ran;
        }


        // WEIBULL (TYPE III EXTREME VALUE) DISTRIBUTION

        // Weibull cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double weibullCDF(double mu, double sigma, double gamma, double upperlimit){
                double arg = (upperlimit - mu)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-Math.pow(arg, gamma));
                return y;
        }

        // Weibull cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double weibullProb(double mu, double sigma, double gamma, double upperlimit){
                double arg = (upperlimit - mu)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-Math.pow(arg, gamma));
                return y;
        }


         // Weibull cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double weibullCDF(double mu, double sigma, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-Math.pow(arg1, gamma));
                if(arg2>=0.0D)term2 = -Math.exp(-Math.pow(arg2, gamma));
                return term2-term1;
        }

        // Weibull cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double weibullProb(double mu, double sigma, double gamma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-Math.pow(arg1, gamma));
                if(arg2>=0.0D)term2 = -Math.exp(-Math.pow(arg2, gamma));
                return term2-term1;
        }

         // Weibull probability
        public static double weibullPDF(double mu,double sigma, double gamma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (gamma/sigma)*Math.pow(arg, gamma-1.0D)*Math.exp(-Math.pow(arg, gamma));
                }
                return y;
        }

        // Weibull probability
        public static double weibull(double mu,double sigma, double gamma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (gamma/sigma)*Math.pow(arg, gamma-1.0D)*Math.exp(-Math.pow(arg, gamma));
                }
                return y;
        }

        // Weibull mean
        public static double weibullMean(double mu,double sigma, double gamma){
                return mu + sigma*Stat.gamma(1.0D/gamma+1.0D);
        }

        // Weibull standard deviation
        public static double weibullStandDev(double sigma, double gamma){
                double y = Stat.gamma(2.0D/gamma+1.0D)-Fmath.square(Stat.gamma(1.0D/gamma+1.0D));
                return sigma*Math.sqrt(y);
        }

        // Weibull mode
        public static double weibullMode(double mu,double sigma, double gamma){
            double y=mu;
            if(gamma>1.0D){
                y = mu + sigma*Math.pow((gamma-1.0D)/gamma, 1.0D/gamma);
            }
            return y;
        }

        // Weibull median
        public static double weibullMedian(double mu,double sigma, double gamma){
            return mu + sigma*Math.pow(Math.log(2.0D),1.0D/gamma);
        }

        // Returns an array of Weibull (Type III EVD) random deviates - clock seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] weibullRand(double mu, double sigma, double gamma, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow(-Math.log(1.0D-rr.nextDouble()),1.0D/gamma)*sigma + mu;
                }
                return ran;
        }

        // Returns an array of Weibull (Type III EVD) random deviates - user supplied seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] weibullRand(double mu, double sigma, double gamma, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow(-Math.log(1.0D-rr.nextDouble()),1.0D/gamma)*sigma + mu;
                }
                return ran;
        }


        // EXPONENTIAL DISTRIBUTION

        // Exponential cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double exponentialCDF(double mu, double sigma, double upperlimit){
                double arg = (upperlimit - mu)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-arg);
                return y;
        }

        // Exponential cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double exponentialProb(double mu, double sigma, double upperlimit){
                double arg = (upperlimit - mu)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-arg);
                return y;
        }

        // Exponential cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double exponentialCDF(double mu, double sigma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-arg1);
                if(arg2>=0.0D)term2 = -Math.exp(-arg2);
                return term2-term1;
        }

        // Exponential cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double exponentialProb(double mu, double sigma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit - mu)/sigma;
                double arg2 = (upperlimit - mu)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-arg1);
                if(arg2>=0.0D)term2 = -Math.exp(-arg2);
                return term2-term1;
        }

        // Exponential probability
        public static double exponentialPDF(double mu,double sigma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = Math.exp(-arg)/sigma;
                }
                return y;
        }

        // Exponential probability
        public static double exponential(double mu,double sigma, double x){
                double arg =(x-mu)/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = Math.exp(-arg)/sigma;
                }
                return y;
        }

        // Exponential mean
        public static double exponentialMean(double mu, double sigma){
                return mu + sigma;
        }

        // Exponential standard deviation
        public static double exponentialStandDev(double sigma){
                return sigma;
        }

        // Exponential mode
        public static double exponentialMode(double mu){
            return mu;
        }

        // Exponential median
        public static double exponentialMedian(double mu,double sigma){
            return mu + sigma*Math.log(2.0D);
        }

        // Returns an array of Exponential random deviates - clock seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] exponentialRand(double mu, double sigma, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = mu - Math.log(1.0D-rr.nextDouble())*sigma;
                }
                return ran;
        }

        // Returns an array of Exponential random deviates - user supplied seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] exponentialRand(double mu, double sigma, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = mu - Math.log(1.0D-rr.nextDouble())*sigma;
                }
                return ran;
        }



        // RAYLEIGH DISTRIBUTION

        // Rayleigh cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double rayleighCDF(double sigma, double upperlimit){
                double arg = (upperlimit)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-arg*arg/2.0D);
                return y;
        }

        // Rayleigh cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double rayleighProb(double sigma, double upperlimit){
                double arg = (upperlimit)/sigma;
                double y = 0.0D;
                if(arg>0.0D)y = 1.0D - Math.exp(-arg*arg/2.0D);
                return y;
        }

        // Rayleigh cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double rayleighCDF(double sigma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit)/sigma;
                double arg2 = (upperlimit)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-arg1*arg1/2.0D);
                if(arg2>=0.0D)term2 = -Math.exp(-arg2*arg2/2.0D);
                return term2-term1;
        }

        // Rayleigh cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double rayleighProb(double sigma, double lowerlimit, double upperlimit){
                double arg1 = (lowerlimit)/sigma;
                double arg2 = (upperlimit)/sigma;
                double term1 = 0.0D, term2 = 0.0D;
                if(arg1>=0.0D)term1 = -Math.exp(-arg1*arg1/2.0D);
                if(arg2>=0.0D)term2 = -Math.exp(-arg2*arg2/2.0D);
                return term2-term1;
        }

        // Rayleigh probability
        public static double rayleighPDF(double sigma, double x){
                double arg =x/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (arg/sigma)*Math.exp(-arg*arg/2.0D)/sigma;
                }
                return y;
        }

        // Rayleigh probability
        public static double rayleigh(double sigma, double x){
                double arg =x/sigma;
                double y = 0.0D;
                if(arg>=0.0D){
                    y = (arg/sigma)*Math.exp(-arg*arg/2.0D)/sigma;
                }
                return y;
        }

        // Rayleigh mean
        public static double rayleighMean(double sigma){
                return sigma*Math.sqrt(Math.PI/2.0D);
        }

        // Rayleigh standard deviation
        public static double rayleighStandDev(double sigma){
                return sigma*Math.sqrt(2.0D-Math.PI/2.0D);
        }

        // Rayleigh mode
        public static double rayleighMode(double sigma){
            return sigma;
        }

        // Rayleigh median
        public static double rayleighMedian(double sigma){
            return sigma*Math.sqrt(Math.log(2.0D));
        }

        // Returns an array of Rayleigh random deviates - clock seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] rayleighRand(double sigma, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = Math.sqrt(-2.0D*Math.log(1.0D-rr.nextDouble()))*sigma;
                }
                return ran;
        }

        // Returns an array of Rayleigh random deviates - user supplied seed
        // mu  =  location parameter, sigma = cale parameter, gamma = shape parametern = length of array
        public static double[] rayleighRand(double sigma, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = Math.sqrt(-2.0D*Math.log(1.0D-rr.nextDouble()))*sigma;
                }
                return ran;
        }


        // PARETO DISTRIBUTION

        // Pareto cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double paretoCDF(double alpha, double beta, double upperlimit){
                double y = 0.0D;
                if(upperlimit>=beta)y = 1.0D - Math.pow(beta/upperlimit, alpha);
                return y;
        }

        // Pareto cumulative distribution function
        // probability that a variate will assume  a value less than the upperlimit
        public static double paretoProb(double alpha, double beta, double upperlimit){
                double y = 0.0D;
                if(upperlimit>=beta)y = 1.0D - Math.pow(beta/upperlimit, alpha);
                return y;
        }

         // Pareto cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double paretoCDF(double alpha, double beta, double lowerlimit, double upperlimit){
                double term1 = 0.0D, term2 = 0.0D;
                if(lowerlimit>=beta)term1 = -Math.pow(beta/lowerlimit, alpha);
                if(upperlimit>=beta)term2 = -Math.pow(beta/upperlimit, alpha);
                return term2-term1;
        }

        // Pareto cumulative distribution function
        // probability that a variate will assume a value between the lower and  the upper limits
        public static double paretoProb(double alpha, double beta, double lowerlimit, double upperlimit){
                double term1 = 0.0D, term2 = 0.0D;
                if(lowerlimit>=beta)term1 = -Math.pow(beta/lowerlimit, alpha);
                if(upperlimit>=beta)term2 = -Math.pow(beta/upperlimit, alpha);
                return term2-term1;
        }

         // Pareto probability
        public static double paretoPDF(double alpha, double beta, double x){
                double y = 0.0D;
                if(x>=beta){
                    y = alpha*Math.pow(beta, alpha)/Math.pow(x, alpha+1.0D);
                }
                return y;
        }

        // Pareto probability
        public static double pareto(double alpha, double beta, double x){
                double y = 0.0D;
                if(x>=beta){
                    y = alpha*Math.pow(beta, alpha)/Math.pow(x, alpha+1.0D);
                }
                return y;
        }

        // Pareto mean
        public static double paretoMean(double alpha, double beta){
                double y = Double.NaN;
                if(alpha>1.0D)y = alpha*beta/(alpha-1);
                return y;
        }

        // Pareto standard deviation
        public static double paretoStandDev(double alpha, double beta){
                double y = Double.NaN;
                if(alpha>1.0D)y = alpha*Fmath.square(beta)/(Fmath.square(alpha-1)*(alpha-2));
                return y;
        }

        // Pareto mode
        public static double paretoMode(double beta){
            return beta;
        }

        // Returns an array of Pareto random deviates - clock seed
        public static double[] paretoRand(double alpha, double beta, int n){
                double[] ran = new double[n];
                Random rr = new Random();
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow(1.0D-rr.nextDouble(), -1.0D/alpha)*beta;
                }
                return ran;
        }

        // Returns an array of Pareto random deviates - user supplied seed
        public static double[] paretoRand(double alpha, double beta, int n, long seed){
                double[] ran = new double[n];
                Random rr = new Random(seed);
                for(int i=0; i<n; i++){
                     ran[i] = Math.pow(1.0D-rr.nextDouble(), -1.0D/alpha)*beta;
                }
                return ran;
        }


        // FITTING DATA TO ABOVE DISTRIBUTIONS

        // Fit a data set to one, several or all of the above distributions (instance)
        public void fitOneOrSeveralDistributions(){
            double[] dd = this.getArray_as_double();
            Regression.fitOneOrSeveralDistributions(dd);
        }

        // Fit a data set to one, several or all of the above distributions (static)
        public static void fitOneOrSeveralDistributions(double[] array){
            Regression.fitOneOrSeveralDistributions(array);
        }


        // OUTLIER TESTING (STATIC)

        // Anscombe test for a upper outlier
        public static Vector<Object> upperOutliersAnscombe(double[] values, double constant){

            Stat am = new Stat(values);
            double[] copy0 = am.getArray_as_double();
            int nValues = values.length;
            int nValues0 = nValues;
            Vector<Object> outers = new Vector<Object>();
            int nOutliers = 0;
            boolean test = true;

            while(test){
                double mean = am.mean_as_double();
                double standDev = am.standardDeviation_as_double();
                double max = am.getMaximum_as_double();
                int maxIndex = am.getMaximumIndex();
                double statistic = (max - mean)/standDev;
                if(statistic>constant){
                    outers.addElement(new Double(max));
                    outers.addElement(new Integer(maxIndex));
                    nOutliers++;
                    double[] copy1 = new double[nValues-1];
                    for(int i=maxIndex; i<nValues-1; i++)copy1[i] = copy0[i+1];

                    nValues--;
                    am = new Stat(copy1.clone());
                }
                else{
                    test=false;
                }
            }

            double[] outliers = null;
            int[] outIndices = null;

            if(nOutliers>0){
                outliers = new double[nOutliers];
                outIndices = new int[nOutliers];
                for(int i=0; i<nOutliers; i++){
                    outliers[i] = ((Double)outers.elementAt(2*i)).doubleValue();
                    outIndices[i] = ((Integer)outers.elementAt(2*i+1)).intValue();
                }
            }

            Vector<Object> ret = new Vector<Object>();
            ret.addElement(new Integer(nOutliers));
            ret.addElement(outliers);
            ret.addElement(outIndices);
            ret.addElement(copy0);
            return ret;
        }

        // Anscombe test for a upper outlier
        public static Vector<Object> upperOutliersAnscombe(BigDecimal[] values, BigDecimal constant){

            Stat am = new Stat(values);
            BigDecimal[] copy0 = am.getArray_as_BigDecimal();
            int nValues = values.length;
            int nValues0 = nValues;
            Vector<Object> outers = new Vector<Object>();
            int nOutliers = 0;
            boolean test = true;
            while(test){
                BigDecimal mean = am.mean_as_BigDecimal();
                BigDecimal variance = am.variance_as_BigDecimal();
                BigDecimal max = am.getMaximum_as_BigDecimal();
                int maxIndex = am.getMaximumIndex();
                BigDecimal statistic = (max.subtract(mean)).divide(variance, BigDecimal.ROUND_HALF_UP);
                if(statistic.compareTo(constant.multiply(constant))==1){
                    outers.addElement(max);
                    outers.addElement(new Integer(maxIndex));
                    nOutliers++;
                    BigDecimal[] copy1 = new BigDecimal[nValues-1];
                    for(int i=maxIndex; i<nValues-1; i++)copy1[i] = copy0[i+1];

                    nValues--;
                    am = new Stat(copy1.clone());
                }
                else{
                    mean = null;
                    variance = null;
                    statistic = null;
                    copy0 = null;
                    test=false;
                }
            }

            BigDecimal[] outliers = null;
            int[] outIndices = null;

            if(nOutliers>0){
                outliers = new BigDecimal[nOutliers];
                outIndices = new int[nOutliers];
                for(int i=0; i<nOutliers; i++){
                    outliers[i] = ((BigDecimal)outers.elementAt(2*i));
                    outIndices[i] = ((Integer)outers.elementAt(2*i+1)).intValue();
                }
            }

            Vector<Object> ret = new Vector<Object>();
            ret.addElement(new Integer(nOutliers));
            ret.addElement(outliers);
            ret.addElement(outIndices);
            ret.addElement(copy0);
            return ret;
        }


        // Anscombe test for a upper outlier
        public Vector<Object> upperOutliersAnscombe(BigInteger[] values, BigInteger constant){
            ArrayMaths am = new ArrayMaths(values);
            BigDecimal[] bd = am.getArray_as_BigDecimal();
            BigDecimal cd = new BigDecimal(constant);
            return Stat.upperOutliersAnscombe(bd, cd);
        }

        // Anscombe test for a lower outlier
        public static Vector<Object> lowerOutliersAnscombe(double[] values, double constant){

            Stat am = new Stat(values);
            double[] copy0 = am.getArray_as_double();
            int nValues = values.length;
            int nValues0 = nValues;
            Vector<Object> outers = new Vector<Object>();
            int nOutliers = 0;
            boolean test = true;

            while(test){
                double mean = am.mean_as_double();
                double standDev = am.standardDeviation_as_double();
                double min = am.getMinimum_as_double();
                int minIndex = am.getMinimumIndex();
                double statistic = (mean - min)/standDev;
                if(statistic>constant){
                    outers.addElement(new Double(min));
                    outers.addElement(new Integer(minIndex));
                    nOutliers++;
                    double[] copy1 = new double[nValues-1];
                    for(int i=minIndex; i<nValues-1; i++)copy1[i] = copy0[i+1];

                    nValues--;
                    am = new Stat(copy1.clone());
                }
                else{
                    test=false;
                }
            }

            double[] outliers = null;
            int[] outIndices = null;

            if(nOutliers>0){
                outliers = new double[nOutliers];
                outIndices = new int[nOutliers];
                for(int i=0; i<nOutliers; i++){
                    outliers[i] = ((Double)outers.elementAt(2*i)).doubleValue();
                    outIndices[i] = ((Integer)outers.elementAt(2*i+1)).intValue();
                }
            }

            Vector<Object> ret = new Vector<Object>();
            ret.addElement(new Integer(nOutliers));
            ret.addElement(outliers);
            ret.addElement(outIndices);
            ret.addElement(copy0);
            return ret;
        }

        // Anscombe test for a lower outlier
        public static Vector<Object> lowerOutliersAnscombe(BigDecimal[] values, BigDecimal constant){

            Stat am = new Stat(values);
            BigDecimal[] copy0 = am.getArray_as_BigDecimal();
            int nValues = values.length;
            int nValues0 = nValues;
            Vector<Object> outers = new Vector<Object>();
            int nOutliers = 0;
            boolean test = true;
            while(test){
                BigDecimal mean = am.mean_as_BigDecimal();
                BigDecimal variance = am.variance_as_BigDecimal();
                BigDecimal min = am.getMinimum_as_BigDecimal();
                int minIndex = am.getMinimumIndex();
                BigDecimal statistic = (mean.subtract(min)).divide(variance, BigDecimal.ROUND_HALF_UP);
                if(statistic.compareTo(constant.multiply(constant))==1){
                    outers.addElement(min);
                    outers.addElement(new Integer(minIndex));
                    nOutliers++;
                    BigDecimal[] copy1 = new BigDecimal[nValues-1];
                    for(int i=minIndex; i<nValues-1; i++)copy1[i] = copy0[i+1];

                    nValues--;
                    am = new Stat(copy1.clone());
                }
                else{
                    mean = null;
                    variance = null;
                    statistic = null;
                    copy0 = null;
                    test=false;
                }
            }

            BigDecimal[] outliers = null;
            int[] outIndices = null;

            if(nOutliers>0){
                outliers = new BigDecimal[nOutliers];
                outIndices = new int[nOutliers];
                for(int i=0; i<nOutliers; i++){
                    outliers[i] = ((BigDecimal)outers.elementAt(2*i));
                    outIndices[i] = ((Integer)outers.elementAt(2*i+1)).intValue();
                }
            }

            Vector<Object> ret = new Vector<Object>();
            ret.addElement(new Integer(nOutliers));
            ret.addElement(outliers);
            ret.addElement(outIndices);
            ret.addElement(copy0);
            return ret;
        }


        // Anscombe test for a lower outlier
        public Vector<Object> lowerOutliersAnscombe(BigInteger[] values, BigInteger constant){
            ArrayMaths am = new ArrayMaths(values);
            BigDecimal[] bd = am.getArray_as_BigDecimal();
            BigDecimal cd = new BigDecimal(constant);
            return Stat.lowerOutliersAnscombe(bd, cd);
        }
}

// CLASSES NEEDED METHODS IN THE ABOVE Stat CLASS

// Class to evaluate the linear correlation coefficient probablity function
// Needed in calls to Integration.gaussQuad
class CorrCoeff implements IntegralFunction{

         public double a;

         public double function(double x){
                  double y = Math.pow((1.0D - x*x),a);
                  return y;
         }
}

// Class to evaluate the Student's t-function
class StudentTfunct implements RealRootFunction{
        public int nu = 0;
        public double cfd = 0.0D;

        public double function(double x){

            double y = cfd - Stat.studentTcdf(x, nu);

            return y;
        }
}

// Class to evaluate the Gamma distribution function
class GammaFunct implements RealRootFunction{
        public double mu = 0.0D;
        public double beta = 0.0D;
        public double gamma = 0.0D;
        public double cfd = 0.0D;

        public double function(double x){

            double y = cfd - Stat.gammaCDF(mu, beta, gamma, x);

            return y;
        }
}

// Class to evaluate the Beta distribution function
class BetaFunct implements RealRootFunction{
        public double alpha = 0.0D;
        public double beta = 0.0D;
        public double min = 0.0D;
        public double max = 0.0D;
        public double cfd = 0.0D;

        public double function(double x){

            double y = cfd - Stat.betaCDF(min, max, alpha, beta, x);

            return y;
        }
}

// Class to evaluate the Erlang B equation
class ErlangBfunct implements RealRootFunction{

         public double blockingProbability = 0.0D;
         public double totalResources = 0.0D;

         public double function(double x){
                return blockingProbability - Stat.erlangBprobability(x, totalResources);
         }
}

// Class to evaluate the Erlang C equation
class ErlangCfunct implements RealRootFunction{

         public double nonZeroDelayProbability = 0.0D;
         public double totalResources = 0.0D;

         public double function(double x){
                return nonZeroDelayProbability - Stat.erlangCprobability(x, totalResources);
         }
}

// Class to evaluate the Engset probability equation
class EngsetProb implements RealRootFunction{

         public double offeredTraffic = 0.0D;
         public double totalResources = 0.0D;
         public double numberOfSources = 0.0D;

         public double function(double x){
                double mTerm = offeredTraffic/(numberOfSources - offeredTraffic*(1.0D - x));
                double pNumer = Stat.logFactorial(numberOfSources-1) - Stat.logFactorial(totalResources) - Stat.logFactorial(numberOfSources-1-totalResources);
                double pDenom = 0.0D;
                double iDenom = 0.0D;
                double iCount = 0.0D;
                double pTerm = 0.0D;
                while(iCount<=totalResources){
                    iDenom = Stat.logFactorial(numberOfSources-1) - Stat.logFactorial(iCount) - Stat.logFactorial(numberOfSources-1-iCount);
                    iDenom += (iCount-totalResources)*Math.log(mTerm);
                    pDenom += Math.exp(iDenom);
                    iCount += 1.0D;
                }
                pTerm = Math.exp(pNumer - Math.log(pDenom));

                return x - pTerm;
         }
}


// Class to evaluate the  Engset load equation
class EngsetLoad implements RealRootFunction{

         public double blockingProbability = 0.0D;
         public double totalResources = 0.0D;
         public double numberOfSources = 0.0D;


         public double function(double x){
                return blockingProbability - Stat.engsetProbability(x, totalResources, numberOfSources);
         }
}

// Class to evaluate the chi-square distribution function
class ChiSquareFunct implements RealRootFunction{

        public double cfd = 0.0D;
        public int nu = 0;

        public double function(double x){

            double y = cfd - Stat.chiSquareCDF(x, nu);

            return y;
        }
}

// Class to evaluate the F-distribution function
class fFunct implements RealRootFunction{

        public double cfd = 0.0D;
        public int nu1 = 0;
        public int nu2 = 0;

        public double function(double x){

            double y = cfd - (1.0D - Stat.fCompCDF(x, nu1, nu2));

            return y;
        }
}

// Class to evaluate the two parameter log-normal distribution function
class LogNormalTwoParFunct implements RealRootFunction{

        public double cfd = 0.0D;
        public double mu = 0.0D;
        public double sigma = 0.0D;

        public double function(double x){

            double y = cfd - Stat.logNormalCDF(mu, sigma, x);

            return y;
        }
}

// Class to evaluate the three parameter log-normal distribution function
class LogNormalThreeParFunct implements RealRootFunction{

        public double cfd = 0.0D;
        public double alpha = 0.0D;
        public double beta = 0.0D;
        public double gamma = 0.0D;

        public double function(double x){

            double y = cfd - Stat.logNormalThreeParCDF(alpha, beta, gamma, x);

            return y;
        }
}